<?php

namespace App\Http\Controllers\Deposit;

use App\Models\BusinessDate;
use Illuminate\Http\Request;
use App\Models\AccountDetails;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use App\Models\MessageRoute;
use App\Models\TransactionDetails;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class DepositController extends Controller
{

    public function depositList()
    {
        $user         = Auth::user();
        $businessDate = BusinessDate::latest('business_date')->first();
        
        $depositListSql = "
            SELECT 
                B.ORG_NAME, 
                CASE A.PAYMENT_TYPE WHEN 'cheque' THEN 'Cheque' WHEN 'po' THEN 'Pay Order' WHEN 'dd' THEN 'Demand Draft' WHEN 'cd' THEN 'Cash Deposit' WHEN 'ft' THEN 'Fund Transfer' ELSE 'N/A' END AS PAYMENT_TYPE, 
                A.CHQ_NO, 
                A.CHQ_DATE, 
                A.AMOUNT, 
                A.BUSINESS_DATE, 
                A.RECORD_DATE 
            FROM 
                IS_TRANSACTION_DETAILS A, 
                GS_MONEY_MKT_ORG B 
            WHERE 
                    A.ORG_ID           = B.ORG_ID(+)
                AND A.TRANSACTION_TYPE = 'deposit'
                and a.PORTFOLIO_CODE   = :investor_code
                AND A.BUSINESS_DATE    = :business_date
            ORDER BY 
                A.TRANSACTION_DETAILS_ID DESC FETCH FIRST 10 ROWS ONLY      
        ";

        $depositListResult = DB::select(
            $depositListSql,
            [
                'business_date' => $businessDate->business_date,
                'investor_code' => $user->investor_code,
            ]
        );

        $deposit_list = [];
        foreach ($depositListResult as $index => $value) {
            $deposit_list[$index]['org_name'] = $value->org_name;
            $deposit_list[$index]['payment_type'] = $value->payment_type;
            $deposit_list[$index]['chq_no'] = $value->chq_no;
            $deposit_list[$index]['amount'] = $value->amount;
            $deposit_list[$index]['chq_date'] = Carbon::createFromFormat('Y-m-d H:i:s', $value->chq_date)->format('d-M-Y');
            $deposit_list[$index]['business_date'] = Carbon::createFromFormat('Y-m-d H:i:s', $value->business_date)->format('d-M-Y');
            $deposit_list[$index]['record_date'] = Carbon::createFromFormat('Y-m-d H:i:s', $value->record_date)->format('d-M-Y H:i:s a');

        }

        return response()->json(
            [
                'status'       => 'success',
                'deposit_list' => $deposit_list,
            ]
        );
    }

    public function submitDeposit(Request $request)
    {
        $validator = Validator::make(
            $request->all(),
            [
                'deposit_amount'        => 'required|numeric',
                    'deposit_account_id'    => 'required|exists:gs_account_details,account_details_id',
                    'payment_type'          => 'required|in:cheque,po,dd,cd,ft',
                    'deposit_instrument_no' => 'required',
                    'bank_id'               => 'required|exists:gs_money_mkt_org,org_id',
                    // 'branch_id'             => 'required|exists:gs_org_branch,org_branch_id',
                    'deposit_date'          => 'required|date_format:' . env('INPUT_DATE_FORMAT'),
                    'deposit_slip'          => 'required|file|extensions:jpg,png,jpeg,pdf',
            ],
            [
                'deposit_amount.required'        => 'The deposit amount filed is required',
                'deposit_amount.numeric'         => 'The deposit amount must be a number',
                'deposit_account_id.required'    => 'The deposit account filed name is required',
                'deposit_account_id.exists'      => 'The deposit account is invalid',
                'payment_type.required'          => 'The payment type filed is required',
                'payment_type.in'                => 'The payment type is invalid',
                'deposit_instrument_no.required' => 'The deposit instrument number filed is required',
                'bank_id.required'               => 'The bank name filed is required',
                'bank_id.exists'                 => 'The bank name is invalid',
                // 'branch_id.required'             => 'The branch name filed is required',
                // 'branch_id.exists'               => 'The branch name is invalid',
                'deposit_date.required'          => 'The deposit instrument date filed is required',
                'deposit_date.date_format'       => 'The deposit instrument date field must be in ' . env('INPUT_DATE_FORMAT') . ' format',
                'deposit_slip.required'          => 'The deposit slip filed is required',
                'deposit_slip.file'              => 'The deposit slip must be a file',
                'deposit_slip.extensions'        => 'The deposit slip must be in jpg, png, jpeg or pdf format',
            ]
        );

        if ($validator->fails()) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Form validation error',
                'errors' => $validator->errors()->messages(),
            ], 400);
        } else {
            try {
                $depositSlipImageName = null;
                $user                 = Auth::user();
                $businessDate         = BusinessDate::latest('business_date')->first();

                if ($request->hasFile('deposit_slip')) {
                    $depositSlipImage     = $request->file('deposit_slip');
                    $depositSlipImageName = 'deposit_slip_'.$user->investor_code.'_'.time().'.'.$depositSlipImage->getClientOriginalExtension();
                    $depositSlipImage->move(public_path().'/upload/deposit_slip', $depositSlipImageName);    
                }


                $transactionDetails                        = new TransactionDetails();
                $transactionDetails->org_id                = $request->bank_id;
                // $transactionDetails->branch_id             = $request->branch_id;
                $transactionDetails->portfolio_code        = $user->investor_code;
                $transactionDetails->transaction_type      = 'deposit';
                $transactionDetails->payment_type          = $request->payment_type;
                $transactionDetails->chq_no                = $request->deposit_instrument_no;
                $transactionDetails->chq_date              = Carbon::createFromFormat(env('INPUT_DATE_FORMAT'), $request->deposit_date)->format('Y-m-d');
                $transactionDetails->amount                = $request->deposit_amount;
                $transactionDetails->remarks               = 'web deposit';
                $transactionDetails->operate_by            = 0;
                $transactionDetails->msg_status            = 'maked';
                $transactionDetails->account_details_id    = $request->deposit_account_id;
                $transactionDetails->file_upload           = $depositSlipImageName;
                $transactionDetails->send_to_bank          = 'n';
                $transactionDetails->honor_dishonor_cancel = 'p';
                $transactionDetails->cheque_print_flag     = 'n';
                $transactionDetails->business_date         = $businessDate->business_date;
                $transactionDetails->record_date           = now();

                if ($transactionDetails->save()) {
                    $messageRoute                         = new MessageRoute();
                    $messageRoute->send_by                = '4772';
                    $messageRoute->receive_by             = '4772';
                    $messageRoute->viewableto             = 4;
                    $messageRoute->transaction_details_id = $transactionDetails->transaction_details_id;
                    $messageRoute->remarks                = 'Deposit by Investor';
                    $messageRoute->status                 = '0';
                    $messageRoute->business_date          = $businessDate->business_date;
                    $messageRoute->record_date            = now();
                    $messageRoute->send_date              = now();

                    $messageRoute->save();

                    return response()->json([
                        'status'      => 'success',
                        'message'   => 'Deposit submitted successfully',
                    ]);
                }
            } catch (\Throwable $th) {
                return response()->json([
                    'status'      => 'error',
                    'dev_message' => $th->getMessage(),
                    'message'      => 'Deposit form submit unsuccessful',
                ], 400);
            }
        }
    }
}
