<?php

namespace App\Http\Controllers\Auth;

use SmsHelper;
use Carbon\Carbon;
use App\Models\Otp;
use App\Models\User;
use App\Models\NomineeInfo;
use App\Models\BusinessDate;
use App\Models\MessageRoute;
use Illuminate\Http\Request;
use Ramsey\Uuid\Type\Integer;
use App\Models\InvestorDetail;
use App\Models\JointApplicant;
use function PHPSTORM_META\map;
use Illuminate\Validation\Rule;
use App\Models\TransactionDetails;
use Illuminate\Support\Facades\DB;

use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Symfony\Component\Console\Input\Input;
use Illuminate\Support\Facades\Mail;

class AuthController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth:api', ['except' => ['login', 'registation', 'signUp', 'forgotPassword', 'operationMode']]);
    }

    public function login(Request $request)
    {
        $request->validate([
            'email'    => 'required|string|email',
            'password' => 'required|string',
        ]);

        $credentials = $request->only('email', 'password');

        $token = Auth::attempt($credentials);

        if (!$token) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Unauthorized',
            ], 401);
        }

        $user = Auth::user();
        return response()->json([
            'status'        => 'success',
            'user'          => $user,
            'authorization' => [
                'token' => $token,
                'type'  => 'bearer',
            ]
        ]);
    }

    public function logout()
    {
        Auth::logout();
        return response()->json([
            'status'  => 'success',
            'message' => 'Successfully logged out',
        ]);
    }

    public function refresh()
    {
        return response()->json([
            'status'        => 'success',
            'user'          => Auth::user(),
            'authorisation' => [
                'token' => Auth::refresh(),
                'type'  => 'bearer',
            ]
        ]);
    }

    public function registation(Request $request, int $pageId)
    {
        try {
            $otpBypass    = true;
            $businessDate = BusinessDate::latest('business_date')->first();

            if ($pageId === 1) {
                $validator = Validator::make(
                    $request->all(),
                    [
                        'email'          => 'required|email|unique:users,email',
                        'contact_number' => 'required|digits:11|unique:users,mobile_no',

                        // 'email'          => 'required|email',
                        // 'contact_number' => 'required|digits:11',
                    ],
                    [
                        'email.required'          => 'The email field is required',
                        'email.email'             => 'The email field is not a valid email address',
                        'email.unique'            => 'The email is already registered',
                        'contact_number.required' => 'The contact number field is required',
                        'contact_number.digits'   => 'The contact number must be 11 digits',
                        'contact_number.unique'   => 'The contact number is already registered',
                    ]
                );

                if ($validator->fails()) {
                    return response()->json([
                        'status'  => 'error',
                        'message' => 'Form validation error',
                        'errors'  => $validator->errors()->messages(),
                    ], 400);
                } else {
                    $investorDetail       = InvestorDetail::with(['nomineeInfo', 'branchDetails'])
                        ->where('email', $request->email)
                        ->orWhere('mobile', $request->contact_number)
                        ->get();

                    if ($investorDetail->count() > 1) {
                        return response()->json([
                            'status'  => 'error',
                            'message' => 'This email or mobile is already registered',
                        ], 400);
                    }
                    
                    $investorDetail = $investorDetail->first();

                    if (!is_null($investorDetail) && $investorDetail->reg_is_complete == 'yes') {
                        return response()->json([
                            'status'  => 'error',
                            'message' => 'This email or mobile is already registered'
                        ], 400);
                    }

                    if ($otpBypass == false) {
                        $otpCode    = rand(123456, 999999);
                        $recordDate = now();
                        $message    = "Your OTP for invesment registration is $otpCode. OTP will be valid for next 3 minutes";

                        $otp = Otp::where('mobile', $request->contact_number)
                            ->where('otp_verify', NULL)
                            ->orderBy('otp_valid_date_time', 'desc')
                            ->first();


                        if ($otp) {
                            if (strtotime($otp->otp_valid_date_time) >= strtotime(now())) {
                                return response()->json([
                                    'status'          => 'success',
                                    'message'         => 'OTP sent to your mobile successfully.'
                                ]);
                            }
                        }

                        $otp                      = new Otp();
                        $otp->message             = $message;
                        $otp->otp_code            = $otpCode;
                        $otp->mobile              = $request->contact_number;
                        $otp->otp_valid_date_time = now()->addMinutes(4);
                        $otp->business_date       = $businessDate->business_date;
                        $otp->record_date         = $recordDate;

                        if ($otp->save()) {
                            $smsHelper = new SmsHelper();
                            $smsResponse = json_decode($smsHelper->sendSingleSms($request->contact_number, $message)->content());
                            // $smsResponse = json_decode($smsHelper->sendSingleSms('01641769553', $message)->content());

                            if ($smsResponse->status == 'error') {
                                $otp->delete();

                                return response()->json([
                                    'status'  => 'error',
                                    'message' => 'We are unable to send OTP for veryfying your request One',
                                    // 'errors' => $smsResponse,
                                ], 400);
                            }

                            $otp->msg_id = $smsResponse->message_info->csms_id;
                            $otp->save();
                        } else {
                            return response()->json([
                                'status'  => 'error',
                                'message' => 'We are unable to send OTP for veryfying your request Two',
                            ], 400);
                        }
                    }

                    return response()->json([
                        'status'          => 'success',
                        'message' => 'OTP sent to your mobile successfully.'
                    ]);
                }
            } elseif ($pageId === 2) {
                $validator = Validator::make(
                    $request->all(),
                    [
                        'otp_code' => 'required|numeric|digits:6',
                    ],
                    [
                        'otp_code.required' => 'The otp field is required',
                        'otp_code.numeric'  => 'The otp field must be a number',
                        'otp_code.digits'   => 'The otp field must be 6 digits',
                    ]
                );

                if ($validator->fails()) {
                    return response()->json([
                        'status'  => 'error',
                        'message' => 'Form validation error',
                        'errors'  => $validator->errors()->messages(),
                    ], 400);
                }

                $otp = Otp::where('otp_code', $request->otp_code)
                    ->where('otp_verify', null)
                    ->where('mobile',$request->contact_number)
                    ->where('otp_valid_date_time', '>=', now())
                    ->first();

                if ($otpBypass == false) {
                    if (is_null($otp)) {
                        return response()->json([
                            'status'     => 'error',
                            'otp_verify' => false,
                            'message' => 'OTP verification unsuccessful'
                        ], 400);
                    }

                    $otp->otp_verify = 'yes';
                    $otp->save();
                }

                $investor_details_arr    = '';
                $joint_details_arr       = '';
                $nominee_details_arr     = [];
                $transaction_details_arr = '';

                $investorDetail       = InvestorDetail::with(['nomineeInfo', 'branchDetails'])
                    ->where('email', $request->email)
                    ->orWhere('mobile', $request->contact_number)
                    ->get();
                $investorDetail = $investorDetail->first();

                if (!is_null($investorDetail)) {
                    $nomineeInfo = $investorDetail->nomineeInfo;
                    $jointInfo   = $investorDetail->jointInvestorDetails;

                    $investor_details_arr = [
                        'investor_details_id'  => $investorDetail->investor_details_id,
                        'investor_name'        => $investorDetail->investor_name,
                        'product_id'           => $investorDetail->template_id,
                        'date_of_birth'        => is_null($investorDetail->dob) ? null : Carbon::createFromFormat('Y-m-d H:i:s', $investorDetail->dob)->format('Y-m-d'),
                        'gender'               => $investorDetail->gender,
                        'father_name'          => $investorDetail->father_name,
                        'mother_name'          => $investorDetail->mother_name,
                        'email'                => $investorDetail->email,
                        'contact_number'       => $investorDetail->mobile,
                        'residency'            => $investorDetail->residency,
                        'nid_number'           => $investorDetail->nid,
                        'occupation_id'        => $investorDetail->occupation_id,
                        'nationality'          => $investorDetail->nationality_id,
                        'present_address'      => $investorDetail->mailing_address,
                        'permanent_address'    => $investorDetail->permanent_address,
                        'bank_id'              => $investorDetail->branchDetails->org_id,
                        'branch_id'            => $investorDetail->org_branch_id,
                        'bank_account_no'      => $investorDetail->bank_account_no,
                        'linked_bo_account_no' => $investorDetail->linked_bo_account_no,
                        'operation_mode'       => $investorDetail->operation_mode_id,
                        'reg_page_no'          => $investorDetail->reg_page_no,
                        'reg_is_complete'      => $investorDetail->reg_is_complete,
                    ];

                    if (!is_null($nomineeInfo)) {
                        foreach ($nomineeInfo as $index => $singleNomineeInfo) {
                            $nominee_details_arr[] = [
                                'nominee_info_id'         => $singleNomineeInfo->nominee_info_id,
                                'nominee_full_name'       => $singleNomineeInfo->nominee_name,
                                'nominee_share'           => $singleNomineeInfo->share_percentage,
                                'nominee_date_of_birth'   => is_null($singleNomineeInfo->dob) ? null : Carbon::createFromFormat('Y-m-d H:i:s', $singleNomineeInfo->dob)->format('Y-m-d'),
                                'nominee_father_name'     => $singleNomineeInfo->father_name,
                                'nominee_mother_name'     => $singleNomineeInfo->mother_name,
                                'nominee_spouse_name'     => $singleNomineeInfo->spouse_name,
                                'nominee_contact_number'  => $singleNomineeInfo->nominee_mobile,
                                'nominee_nid'             => $singleNomineeInfo->nid,
                                'nominee_mailing_address' => $singleNomineeInfo->mailing_address,
                                'nominee_occupation_id'   => $singleNomineeInfo->occupation_id,
                                'nominee_relationship_id' => $singleNomineeInfo->relation_id,
                                'nominee_email'           => $singleNomineeInfo->email,
                            ];
                        }
                    }

                    if(!is_null($jointInfo)){
                        $joint_details_arr=[
                            'joint_applicant_id'   => $jointInfo->joint_applicant_id,
                            'ja_relationship_id'   => $jointInfo->relation_id,
                            'ja_nationality'       => $jointInfo->nationality_id,
                            'ja_occupation_id'     => $jointInfo->occupation_id,
                            'ja_bank_id'           => $jointInfo->branchDetails->org_id,
                            'ja_branch_id'         => $jointInfo->org_branch_id,
                            'ja_bank_account_no'   => $jointInfo->bank_account_no,
                            'ja_investor_name'     => $jointInfo->applicant_name,
                            'ja_father_name'       => $jointInfo->father_name,
                            'ja_mother_name'       => $jointInfo->mother_name,
                            'ja_spouse_name'       => $jointInfo->spouse_name,
                            'ja_residency'         => $jointInfo->residency,
                            'ja_nid_number'        => $jointInfo->nid,
                            'ja_date_of_birth'     => is_null($jointInfo->dob) ? null : Carbon::createFromFormat('Y-m-d H:i:s', $jointInfo->dob)->format('Y-m-d'),
                            'ja_present_address'   => $jointInfo->mailing_address,
                            'ja_permanent_address' => $jointInfo->permanent_address,
                            'ja_contact_number'    => $jointInfo->mobile,
                            'ja_email'             => $jointInfo->email
                        ];
                    }

                    $transactionDetails = TransactionDetails::where('investor_details_id', $investorDetail->investor_details_id)->first();

                    if($transactionDetails){
                        $transaction_details_arr=[
                            'deposit_account_id'    => $transactionDetails->account_details_id,
                            'bank_id'               => $transactionDetails->org_id,
                            'deposit_date'          => is_null($transactionDetails->chq_date) ? null : Carbon::createFromFormat('Y-m-d H:i:s', $transactionDetails->chq_date)->format('Y-m-d'),
                            'deposit_instrument_no' => $transactionDetails->chq_no,
                            'payment_type'          => $transactionDetails->payment_type,
                            'deposit_amount'        => $transactionDetails->amount
                        ];
                    }
                }


                return response()->json([
                    'status'             => 'success',
                    'otp_verify'         => true,
                    'investor_detail'    => $investor_details_arr,
                    'nominee_detail'     => $nominee_details_arr,
                    'joint_detail'       => $joint_details_arr,
                    'transaction_detail' => $transaction_details_arr,
                    'message'            => 'OTP verification successful'
                ]);

            } elseif ($pageId === 3) {
                $validator = Validator::make(
                    $request->all(),
                    [
                        'investor_details_id'    => 'nullable',
                        'investor_name'          => 'required',
                        'product_id'             => "required|exists:ls_template,template_id",
                        'date_of_birth'          => 'required|date_format:Y-m-d',
                        'gender'                 => 'required|in:Male,Female',
                        'father_name'            => 'required',
                        'mother_name'            => 'required',
                        'email'                  => 'required|email|unique:users,email',
                        'contact_number'         => 'required|unique:users,mobile_no',
                        'residency'              => 'required|in:resident,nonresident',
                        'nid_number'             => ['required', 'regex:/^(?:\d{10}|\d{17})$/'],
                        'occupation_id'          => 'required|exists:gs_occupation,occupation_id',
                        'nationality'            => 'required|exists:gs_nationality,nationality_id',
                        'present_address'        => 'required',
                        'permanent_address'      => 'required',
                        'bank_id'                => 'required|exists:gs_money_mkt_org,org_id',
                        'branch_id'              => 'required|exists:gs_org_branch,org_branch_id',
                        'bank_account_no'        => 'required',
                        'operation_mode'         => 'required',
                        'link_bo'                => 'nullable',
                        'link_bo_account_number' => [
                            Rule::requiredIf(function () use ($request) {
                                return in_array($request->get('link_bo'), ['checked']);
                            }),
                            Rule::when(
                                function () use ($request) {
                                    return in_array($request->get('link_bo'), ['checked']);
                                },
                                'digits:16'
                            )
                        ],

                        'ja_investor_name'  => 'required_if:operation_mode,==,2|nullable',
                        'ja_father_name'    => 'required_if:operation_mode,==,2|nullable',
                        'ja_mother_name'    => 'required_if:operation_mode,==,2|nullable',
                        'ja_date_of_birth'  => 'required_if:operation_mode,==,2|nullable|date_format:Y-m-d',
                        'ja_email'          => 'required_if:operation_mode,==,2|nullable|email',
                        'ja_contact_number' => 'required_if:operation_mode,==,2|nullable',
                        'ja_residency'      => 'required_if:operation_mode,==,2|nullable|in:resident,nonresident',
                        // 'ja_nid_number'        => ['required_if:operation_mode,==,2','nullable', 'regex:/^(?:\d{10}|\d{17})$/'],
                        'ja_nid_number'        => ['required_if:operation_mode,2', 'nullable', 'regex:/^(?:\d{10}|\d{17})$/'],
                        'ja_occupation_id'     => 'required_if:operation_mode,==,2|nullable|exists:gs_occupation,occupation_id',
                        'ja_nationality'       => 'required_if:operation_mode,==,2|nullable|exists:gs_nationality,nationality_id',
                        'ja_present_address'   => 'nullable',
                        'ja_permanent_address' => 'nullable',
                        'ja_bank_id'           => 'required_if:operation_mode,==,2|nullable|exists:gs_money_mkt_org,org_id',
                        'ja_branch_id'         => 'required_if:operation_mode,==,2|nullable|exists:gs_org_branch,org_branch_id',
                        'ja_bank_account_no'   => 'required_if:operation_mode,==,2|nullable',
                        'ja_relationship_id'   => 'required_if:operation_mode,==,2|nullable|exists:gs_relation,relation_id',
                    ],
                    [
                        // 'investor_details_id.required'      => 'The investor id is required',
                        'investor_name.required'             => 'The investor name field is required',
                        'product_id.required'                => 'The product field is required',
                        'product_id.exists'                  => 'The product field is invalid',
                        'date_of_birth.required'             => 'The investor date of birth field is required',
                        'date_of_birth.date_format'          => 'The investor date of birth field must be in Y-m-d format',
                        'gender.required'                    => 'The gender from field is required',
                        'gender.in'                          => 'The gender from field is invalid',
                        'father_name.required'               => 'The father name field is required',
                        'mother_name.required'               => 'The mother name field is required',
                        'email.required'                     => 'The email field is required',
                        'email.email'                        => 'The email field is not a valid email address',
                        'email.unique'                       => 'The email is already registered',
                        'residency.required'                 => 'The residency field is required',
                        'residency.in'                       => 'The residency field is invalid',
                        'nid_number.required'                => 'The nid field is required',
                        'nid_number.regex'                   => 'The nid number must be either 10 or 17 digit',
                        'occupation_id.required'             => 'The occupation field is required',
                        'occupation_id.exists'               => 'The occupation field is invalid',
                        'nationality.required'               => 'The nationality field is required',
                        'nationality.exists'                 => 'The nationality field is invalid',
                        'bank_id.required'                   => 'The bank field is required',
                        'bank_id.exists'                     => 'The bank field is invalid',
                        'branch_id.required'                 => 'The branch field is required',
                        'branch_id.exists'                   => 'The branch field is invalid',
                        'bank_account_no.required'           => 'The bank account no field is required',
                        'present_address.required'           => 'The present address field is required',
                        'permanent_address.required'         => 'The permanent address field is required',
                        'link_bo_account_number.required_if' => 'The applicant linked BO account no field is required',
                        'link_bo_account_number.digits'      => 'The applicant linked BO account field must be 16 digits',

                        'ja_investor_name.required_if'   => 'The joint applicant name field is required',
                        'ja_date_of_birth.required_if'   => 'The joint applicant date of birth field is required',
                        'ja_date_of_birth.date_format'   => 'The joint applicant date of birth field must be in Y-m-d format',
                        'ja_gender.required_if'          => 'The joint applicantgender from field is required',
                        'ja_gender.in'                   => 'The joint applicant gender from field is invalid',
                        'ja_father_name.required_if'     => 'The joint applicant father name field is required',
                        'ja_mother_name.required_if'     => 'The joint applicant mother name field is required',
                        'ja_email.required_if'           => 'The joint applicant email field is required',
                        'ja_email.email'                 => 'The joint applicant email field is not a valid email address',
                        'ja_residency.required_if'       => 'The joint applicant residency field is required',
                        'ja_residency.in'                => 'The joint applicant residency field is invalid',
                        'ja_nid_number.required_if'      => 'The joint applicant nid field is required',
                        'ja_nid_number.regex'            => 'The joint applicant nid number must be either 10 or 17 digit',
                        'ja_occupation_id.required_if'   => 'The joint applicant occupation field is required',
                        'ja_occupation_id.exists'        => 'The joint applicant occupation field is invalid',
                        'ja_nationality.required_if'     => 'The joint applicant nationality field is required',
                        'ja_nationality.exists'          => 'The joint applicant nationality field is invalid',
                        'ja_bank_id.required_if'         => 'The joint applicant bank field is required',
                        'ja_bank_id.exists'              => 'The joint applicant bank field is invalid',
                        'ja_branch_id.required_if'       => 'The joint applicant branch field is required',
                        'ja_branch_id.exists'            => 'The joint applicant branch field is invalid',
                        'ja_bank_account_no.required_if' => 'The joint applicant bank account no field is required',
                    ]
                );

                if ($validator->fails()) {
                    return response()->json([
                        'status'  => 'error',
                        'message' => 'Form validation error',
                        'errors'  => $validator->errors()->messages(),
                        // 'request_data'  => $request->all(),
                    ], 400);
                }

                // dd($request->all());

                $investorDetail = InvestorDetail::where('email', $request->email)
                    ->orWhere('mobile', $request->contact_number)
                    ->first();

                // dd($investorDetail);

                if (!is_null($investorDetail) && $investorDetail->reg_is_complete == 'yes') {
                    return response()->json([
                        'status'  => 'error',
                        'message' => 'This email or mobile is already registered',
                        // 'errors'  => $validator->errors()->messages(),
                        // 'request_data'  => $request->all(),
                    ], 400);
                }

                // $investorDetail = InvestorDetail::where('investor_details_id', $request->investor_details_id)
                // ->where('reg_is_complete' , '=', null)
                // ->first();


                if (is_null($investorDetail)) {
                    $investorDetail = new InvestorDetail();
                }

                $investorDetail->org_branch_id        = $request->branch_id;
                $investorDetail->bank_account_no      = $request->bank_account_no;
                $investorDetail->branch_id            = 41;
                $investorDetail->operation_mode_id    = $request->operation_mode;
                $investorDetail->investor_name        = $request->investor_name;
                $investorDetail->template_id          = $request->product_id;
                $investorDetail->dob                  = Carbon::createFromFormat('Y-m-d', $request->date_of_birth)->format('Y-m-d H:i:s');
                $investorDetail->gender               = $request->gender;
                $investorDetail->father_name          = $request->father_name;
                $investorDetail->mother_name          = $request->mother_name;
                $investorDetail->email                = $request->email;
                $investorDetail->mobile               = $request->contact_number;
                $investorDetail->residency            = $request->residency;
                $investorDetail->nid                  = $request->nid_number;
                $investorDetail->occupation_id        = $request->occupation_id;
                $investorDetail->nationality_id       = $request->nationality;
                $investorDetail->mailing_address      = $request->present_address;
                $investorDetail->permanent_address    = $request->permanent_address;
                $investorDetail->linked_bo_account_no = $request->link_bo_account_number;
                $investorDetail->msg_status           = 'pending';
                $investorDetail->reg_page_no          = 1;
                $investorDetail->business_date        = $businessDate->business_date;
                $investorDetail->record_date          = now();
                $investorDetail->save();

                JointApplicant::where('investor_details_id', $investorDetail->investor_details_id)
                    // ->where('joint_applicant_id', $request->joint_applicant_id)
                    ->delete();

                if ((int)$request->operation_mode == 2) {
                    $jointApplicantDetail                      = new JointApplicant();
                    $jointApplicantDetail->investor_details_id = $investorDetail->investor_details_id;
                    $jointApplicantDetail->relation_id         = $request->ja_relationship_id;
                    $jointApplicantDetail->nationality_id      = $request->ja_nationality;
                    $jointApplicantDetail->occupation_id       = $request->ja_occupation_id;
                    $jointApplicantDetail->org_branch_id       = $request->ja_branch_id;
                    $jointApplicantDetail->applicant_name      = $request->ja_investor_name;
                    $jointApplicantDetail->father_name         = $request->ja_father_name;
                    $jointApplicantDetail->mother_name         = $request->ja_mother_name;
                    $jointApplicantDetail->spouse_name         = $request->ja_spouse_name;
                    $jointApplicantDetail->residency           = $request->ja_residency;
                    $jointApplicantDetail->nid                 = $request->ja_nid_number;
                    // $jointApplicantDetail->bo_account_no       = $request->ja_bo_no;
                    $jointApplicantDetail->dob               = Carbon::createFromFormat('Y-m-d', $request->ja_date_of_birth)->format('Y-m-d H:i:s');
                    $jointApplicantDetail->bank_account_no   = $request->ja_bank_account_no;
                    $jointApplicantDetail->mailing_address   = $request->ja_present_address;
                    $jointApplicantDetail->permanent_address = $request->ja_permanent_address;
                    $jointApplicantDetail->mobile            = $request->ja_contact_number;
                    $jointApplicantDetail->email             = $request->ja_email;
                    $jointApplicantDetail->business_date     = $businessDate->business_date;
                    $jointApplicantDetail->record_date       = now();
                    $jointApplicantDetail->save();
                }

                return response()->json([
                    'status'              => 'success',
                    'investor_details_id' => $investorDetail->investor_details_id,
                    'joint_applicant_id'  => isset($jointApplicantDetail) ? $jointApplicantDetail->joint_applicant_id : null,
                ]);
            } elseif ($pageId === 4) {
                $requestData = $request->all();
                $validator   = Validator::make(
                    $requestData,
                    [
                        'investor_details_id' => 'required|exists:is_investor_details,investor_details_id',
                        // 'nominee_info_id.*'         => 'nullable',
                        'nominee_full_name.*'       => 'required',
                        'nominee_share.*'           => 'required|numeric',
                        'nominee_date_of_birth.*'   => 'required|date_format:Y-m-d',
                        // 'nominee_father_name.*'     => 'required',
                        // 'nominee_mother_name.*'     => 'required',
                        'nominee_contact_number.*'  => 'required',
                        'nominee_nid.*'             => 'required',
                        // 'nominee_occupation_id.*'   => 'exists:gs_occupation,occupation_id',
                        'nominee_relationship_id.*' => 'required|exists:gs_relation,relation_id',
                    ],
                    [
                        'investor_details_id.required'        => "The investor details id is required",
                        'investor_details_id.exists'          => "The investor details id is invalid",
                        'nominee_full_name.*.required'        => "The nominee full name field is required",
                        'nominee_share.*.required'            => "The nominee share field is required",
                        'nominee_share.*.numeric'             => "The nominee share field must be a number",
                        'nominee_date_of_birth.*.required'    => "The nominee date of birth field is required",
                        'nominee_date_of_birth.*.date_format' => "The nominee date of birth field must be in Y-m-d format",
                        // 'nominee_father_name.*.required'      => "The nominee father name field is required",
                        // 'nominee_mother_name.*.required'      => "The nominee mother name field is required",
                        'nominee_contact_number.*.required'   => "The nominee contact number field is required",
                        'nominee_nid.*.required'              => "The nominee nid field is required",
                        // 'nominee_occupation_id.*.required'    => "The nominee occupation field is required",
                        // 'nominee_occupation_id.*.exists'      => "The nominee occupation field is invalid",
                        'nominee_relationship_id.*.required'  => "The nominee relationship field is required",
                        'nominee_relationship_id.*.exists'    => "The nominee relationship field is invalid",
                    ]
                );


                if ($validator->fails()) {
                    $errors = [];
                    foreach ($validator->errors()->messages() as $fieldName => $messages) {
                        $fieldNamePieces = explode('.', $fieldName);
                        // $errors[$fieldNamePieces[0]][$fieldNamePieces[1]] = $messages[0];
                        $errors[$fieldNamePieces[1]][$fieldNamePieces[0]] = $messages[0];
                    }

                    ksort($errors);

                    return response()->json([
                        'status'  => 'error',
                        'message' => 'Form validation error',
                        'errors'  => $errors,
                    ], 400);
                }

                $length        = (count($requestData['nominee_full_name']) < 3) ?  count($requestData['nominee_full_name']) : 2;
                $nomineeInfoId = [];
                NomineeInfo::where('investor_details_id', $request->investor_details_id)->delete();

                for ($i = 0; $i < $length; $i++) {
                    try {
                        // if (is_null($requestData['nominee_full_name'][$i])) {
                        //     $nomineeInfo = new NomineeInfo();
                        // } else {
                        //     $nomineeInfo = NomineeInfo::where('investor_details_id', $request->investor_details_id)
                        //     ->where('nominee_info_id', $requestData['nominee_info_id'][$i])
                        //     ->first();

                        //     if (is_null($nomineeInfo)) {
                        //         $nomineeInfo = new NomineeInfo();
                        //     }
                        // }

                        $nomineeInfo = new NomineeInfo();

                        $nomineeInfo->investor_details_id = $request->investor_details_id;
                        $nomineeInfo->nominee_name        = $requestData['nominee_full_name'][$i];
                        $nomineeInfo->share_percentage    = $requestData['nominee_share'][$i];
                        $nomineeInfo->dob                 = Carbon::createFromFormat('Y-m-d', $requestData['nominee_date_of_birth'][$i])->format('Y-m-d H:i:s');
                        $nomineeInfo->father_name         = $requestData['nominee_father_name'][$i];
                        $nomineeInfo->mother_name         = $requestData['nominee_mother_name'][$i];
                        $nomineeInfo->spouse_name         = $requestData['nominee_spouse_name'][$i];
                        $nomineeInfo->nominee_mobile      = $requestData['nominee_contact_number'][$i];
                        $nomineeInfo->email               = $requestData['nominee_email'][$i];
                        $nomineeInfo->nid                 = $requestData['nominee_nid'][$i];
                        $nomineeInfo->mailing_address     = $requestData['nominee_mailing_address'][$i];
                        $nomineeInfo->occupation_id       = $requestData['nominee_occupation_id'][$i];
                        $nomineeInfo->relation_id         = $requestData['nominee_relationship_id'][$i];
                        $nomineeInfo->business_date       = $businessDate->business_date;
                        $nomineeInfo->record_date         = now();
                        $nomineeInfo->save();

                        $nomineeInfoId[] = $nomineeInfo->nominee_info_id;
                    } catch (\Throwable $th) {
                        //throw $th;
                    }

                    $investorDetail              = InvestorDetail::find($request->investor_details_id);
                    $investorDetail->reg_page_no = 2;
                }

                $investorDetail->save();

                return response()->json([
                    'status'              => 'success',
                    'investor_details_id' => $request->investor_details_id,
                    'nominee_info_id'     => $nomineeInfoId,
                ]);
            } elseif ($pageId === 5) {
                $validator = Validator::make(
                    $request->all(),
                    [
                        'investor_details_id'   => 'required|exists:is_investor_details,investor_details_id',
                        'deposit_amount'        => 'required|numeric',
                        'deposit_account_id'    => 'required|exists:gs_account_details,account_details_id',
                        'payment_type'          => 'required|in:cheque,po,dd,ft,cd,bkash',
                        'deposit_instrument_no' => 'required',
                        'bank_id'               => 'required_if:payment_type,==,cheque|required_if:payment_type,==,po|required_if:payment_type,==,dd|required_if:payment_type,==,ft|nullable|exists:gs_money_mkt_org,org_id',
                        'deposit_date'          => 'required|date_format:Y-m-d',
                        'deposit_slip'          => 'required|file|extensions:jpg,png,jpeg,pdf|max:5120',
                    ],
                    [
                        'investor_details_id.required'   => "The investor details id is required",
                        'investor_details_id.exists'     => "The investor details id is invalid",
                        'deposit_amount.required'        => 'The deposit amount filed is required',
                        'deposit_amount.numeric'         => 'The deposit amount must be a number',
                        'deposit_account_id.required'    => 'The deposit account filed name is required',
                        'deposit_account_id.exists'      => 'The deposit account is invalid',
                        'payment_type.required'          => 'The payment type filed is required',
                        'payment_type.in'                => 'The payment type is invalid',
                        'deposit_instrument_no.required' => 'The deposit instrument number filed is required',
                        'bank_id.required_if'            => 'The bank name filed is required when payment type is not cash deposit or bkash',
                        'bank_id.required'               => 'The bank name filed is required',
                        'bank_id.exists'                 => 'The bank name is invalid',
                        'deposit_date.required'          => 'The deposit instrument date filed is required',
                        'deposit_date.date_format'       => 'The deposit instrument date field must be in Y-m-d format',
                        'deposit_slip.required'          => 'The deposit slip filed is required',
                        'deposit_slip.file'              => 'The deposit slip must be a file',
                        'deposit_slip.extensions'        => 'The deposit slip must be in jpg, png, jpeg or pdf format',
                    ]
                );

                if ($validator->fails()) {
                    return response()->json([
                        'status'  => 'error',
                        'message' => 'Form validation error',
                        'errors'  => $validator->errors()->messages(),
                    ], 400);
                }

                // return response()->json([
                //     'status'  => 'error',
                //     'message' => 'Form validation error',
                //     'request'  => $request->all(),
                // ], 200);

                try {
                    $depositSlipImageName = null;
                    $businessDate         = BusinessDate::latest('business_date')->first();

                    if ($request->hasFile('deposit_slip')) {
                        $depositSlipImage     = $request->file('deposit_slip');
                        $depositSlipImageName = 'deposit_slip_' . $request->investor_details_id . '_' . time() . '.' . $depositSlipImage->getClientOriginalExtension();
                        $depositSlipImage->move(public_path() . '/upload/deposit_slip', $depositSlipImageName);
                    }

                    $transactionDetails = TransactionDetails::where('investor_details_id', $request->investor_details_id)->first();

                    if (is_null($transactionDetails)) {
                        $transactionDetails = new TransactionDetails();
                    }

                    $transactionDetails->org_id                = $request->bank_id;
                    $transactionDetails->investor_details_id   = $request->investor_details_id;
                    $transactionDetails->transaction_type      = 'deposit';
                    $transactionDetails->payment_type          = $request->payment_type;
                    $transactionDetails->chq_no                = $request->deposit_instrument_no;
                    $transactionDetails->chq_date              = Carbon::createFromFormat('Y-m-d', $request->deposit_date)->format('Y-m-d H:i:s');
                    $transactionDetails->amount                = $request->deposit_amount;
                    $transactionDetails->remarks               = 'web deposit';
                    $transactionDetails->operate_by            = 0;
                    $transactionDetails->msg_status            = '';
                    $transactionDetails->account_details_id    = $request->deposit_account_id;
                    $transactionDetails->file_upload           = $depositSlipImageName;
                    $transactionDetails->send_to_bank          = 'n';
                    $transactionDetails->honor_dishonor_cancel = 'p';
                    $transactionDetails->cheque_print_flag     = 'n';
                    $transactionDetails->business_date         = $businessDate->business_date;
                    $transactionDetails->record_date           = now();

                    if ($transactionDetails->save()) {
                        $messageRoute                         = new MessageRoute();
                        $messageRoute->send_by                = '4772';
                        $messageRoute->receive_by             = '4772';
                        $messageRoute->viewableto             = 4;
                        $messageRoute->transaction_details_id = $transactionDetails->transaction_details_id;
                        $messageRoute->remarks                = 'Deposit by Investor';
                        $messageRoute->status                 = '0';
                        $messageRoute->business_date          = $businessDate->business_date;
                        $messageRoute->record_date            = now();
                        $messageRoute->send_date              = now();

                        $messageRoute->save();

                        return response()->json([
                            'status'  => 'success',
                            'message' => 'Deposit submitted successfully',
                        ]);
                    }
                } catch (\Throwable $th) {
                    return response()->json([
                        'status'      => 'error',
                        'message'     => 'Deposit form submit unsuccessful',
                        'dev_message' => $th->getMessage(),
                    ], 400);
                }
            } elseif ($pageId === 6) {
                /**
                 * insert file for investor and nominee
                 */

                $investorDetail = null;
                $rules          = [
                    'investor_details_id'     => 'required|exists:is_investor_details,investor_details_id',
                    'investor_photo'          => 'required|file|extensions:jpg,png,jpeg|max:5120',
                    'investor_nid_photo'      => 'required|file|extensions:jpg,png,jpeg|max:5120',
                    'investor_nid_back_photo' => 'required|file|extensions:jpg,png,jpeg|max:5120',
                    'investor_signature'      => 'required|file|extensions:jpg,png,jpeg|max:5120',
                    'investor_cheque_photo'   => 'required|file|extensions:jpg,png,jpeg|max:5120',
                ];

                if (!is_null($request->investor_details_id)) {
                    $investorDetail = InvestorDetail::find($request->investor_details_id);

                    if (!is_null($investorDetail->linked_bo_account_no) && is_null($investorDetail->reg_is_complete)) {
                        $rules['link_bo_photo'] = 'required|mimes:jpg,jpeg,png,bmp|file|extensions:jpg,png,jpeg|max:5120';
                    }

                    if ($investorDetail->operation_mode_id == 2 && is_null($investorDetail->reg_is_complete)) {
                        $jointApplicantDetail           = $investorDetail->jointInvestorDetails;
                        $rules['joint_applicant_photo']    = 'required|mimes:jpg,jpeg,png,bmp|file|extensions:jpg,png,jpeg|max:5120';
                        $rules['joint_applicant_sign']     = 'required|mimes:jpg,jpeg,png,bmp|file|extensions:jpg,png,jpeg|max:5120';
                        $rules['joint_applicant_nid']      = 'required|mimes:jpg,jpeg,png,bmp|file|extensions:jpg,png,jpeg|max:5120';
                        $rules['joint_applicant_nid_back'] = 'required|mimes:jpg,jpeg,png,bmp|file|extensions:jpg,png,jpeg|max:5120';

                        if (!is_null($jointApplicantDetail->bo_account_no)) {
                            // $rules['joint_bo_photo'] = 'required|mimes:jpg,jpeg,png,bmp|file|extensions:jpg,png,jpeg';
                        }
                    }
                }

                // dd('Test');

                $fileCounterArr        = [];
                $nomineeInfoIds        = $request->get('nominee_info_id');
                $nomineePhotoFiles     = $request->file('nominee_photo');
                $nomineeNidPhotoFiles  = $request->file('nominee_nid_photo');
                $nomineeSignatureFiles = $request->file('nominee_signature');

                if (!is_null($nomineeInfoIds)) {
                    // dd($nomineeInfoIds, array_filter($nomineeInfoIds), array_keys($nomineeInfoIds), array_keys(array_filter($nomineeInfoIds)));
                    $fileCounterArr[] = max(array_keys(array_filter($nomineeInfoIds)));
                }

                if (!is_null($nomineePhotoFiles)) {
                    $fileCounterArr[] = max(array_keys($nomineePhotoFiles));
                }

                if (!is_null($nomineeNidPhotoFiles)) {
                    $fileCounterArr[] = max(array_keys($nomineeNidPhotoFiles));
                }

                if (!is_null($nomineeSignatureFiles)) {
                    $fileCounterArr[] = max(array_keys($nomineeSignatureFiles));
                }

                for ($i = 0; $i <= max($fileCounterArr); $i++) {
                    $rules['nominee_info_id.' . $i]        = 'required|exists:is_nominee_info,nominee_info_id';
                    $rules['nominee_photo.' . $i]          = 'required|mimes:jpg,jpeg,png,bmp|file|extensions:jpg,png,jpeg|max:5120';
                    $rules['nominee_nid_photo.' . $i]      = 'required|mimes:jpg,jpeg,png,bmp|file|extensions:jpg,png,jpeg|max:5120';
                    $rules['nominee_nid_back_photo.' . $i] = 'required|mimes:jpg,jpeg,png,bmp|file|extensions:jpg,png,jpeg|max:5120';
                    $rules['nominee_signature.' . $i]      = 'required|mimes:jpg,jpeg,png,bmp|file|extensions:jpg,png,jpeg|max:5120';
                }

                if (count($fileCounterArr) == 0) {
                    $rules['nominee_info_id.' . count($fileCounterArr)]        = 'required|exists:is_nominee_info,nominee_info_id';
                    $rules['nominee_photo.' . count($fileCounterArr)]          = 'required|mimes:jpg,jpeg,png,bmp|file|extensions:jpg,png,jpeg|max:5120';
                    $rules['nominee_nid_photo.' . count($fileCounterArr)]      = 'required|mimes:jpg,jpeg,png,bmp|file|extensions:jpg,png,jpeg|max:5120';
                    $rules['nominee_nid_back_photo.' . count($fileCounterArr)] = 'required|mimes:jpg,jpeg,png,bmp|file|extensions:jpg,png,jpeg|max:5120';
                    $rules['nominee_signature.' . count($fileCounterArr)]      = 'required|mimes:jpg,jpeg,png,bmp|file|extensions:jpg,png,jpeg|max:5120';
                }

                $validator = Validator::make(
                    $request->all(),
                    $rules,
                    [
                        'investor_details_id.required'  => "The investor details id is required",
                        'investor_details_id.exists'    => "The investor details id is invalid",

                        'investor_photo.required'       => 'The investor photo filed is required',
                        'investor_photo.file'           => 'The investor photo must be a file',
                        'investor_photo.extensions'     => 'The investor photo must be in jpg, png or jpeg format',
                        'investor_photo.max'            => 'The investor photo must less than 5 mb',

                        'investor_nid_photo.required'   => 'The investor nid photo filed is required',
                        'investor_nid_photo.file'       => 'The investor nid photo must be a file',
                        'investor_nid_photo.extensions' => 'The investor nid photo must be in jpg, png or jpeg format',
                        'investor_nid_photo.max'        => 'The investor nid photo must less than 5 mb',

                        'investor_nid_back_photo.required'   => 'The investor nid photo filed is required',
                        'investor_nid_back_photo.file'       => 'The investor nid photo must be a file',
                        'investor_nid_back_photo.extensions' => 'The investor nid photo must be in jpg, png or jpeg format',
                        'investor_nid_back_photo.max'        => 'The investor nid photo must less than 5 mb',

                        'investor_signature.required'   => 'The investor signature filed is required',
                        'investor_signature.file'       => 'The investor signature must be a file',
                        'investor_signature.extensions' => 'The investor signature must be in jpg, png or jpeg format',
                        'investor_signature.max'        => 'The investor signature photo must less than 5 mb',

                        'investor_cheque_photo.required'   => 'The investor cheque photo filed is required',
                        'investor_cheque_photo.file'       => 'The investor cheque photo must be a file',
                        'investor_cheque_photo.extensions' => 'The investor cheque photo must be in jpg, png or jpeg format',
                        'investor_cheque_photo.max'        => 'The investor cheque photo must less than 5 mb',


                        'joint_applicant_photo.required'       => 'The joint investor photo filed is required',
                        'joint_applicant_photo.file'           => 'The joint investor photo must be a file',
                        'joint_applicant_photo.extensions'     => 'The joint investor photo must be in jpg, png or jpeg format',
                        'joint_applicant_photo.max'            => 'The joint investor photo must less than 5 mb',

                        'joint_applicant_nid.required'   => 'The joint investor nid photo filed is required',
                        'joint_applicant_nid.file'       => 'The joint investor nid photo must be a file',
                        'joint_applicant_nid.extensions' => 'The joint investor nid photo must be in jpg, png or jpeg format',
                        'joint_applicant_nid.max'        => 'The joint investor nid photo must less than 5 mb',

                        'joint_applicant_nid_back.required'   => 'The joint investor nid photo filed is required',
                        'joint_applicant_nid_back.file'       => 'The joint investor nid photo must be a file',
                        'joint_applicant_nid_back.extensions' => 'The joint investor nid photo must be in jpg, png or jpeg format',
                        'joint_applicant_nid_back.max'        => 'The joint investor nid photo must less than 5 mb',

                        'joint_applicant_sign.required'   => 'The joint investor signature filed is required',
                        'joint_applicant_sign.file'       => 'The joint investor signature must be a file',
                        'joint_applicant_sign.extensions' => 'The joint investor signature must be in jpg, png or jpeg format',
                        'joint_applicant_sign.max'        => 'The joint investor signature photo must less than 5 mb',

                        'nominee_info_id.*.required' => "The nominee info id is required",
                        'nominee_info_id.*.exists'   => "The nominee info id is invalid",
                        'nominee_photo.*.required'   => 'The nominee photo filed is required',
                        'nominee_photo.*.file'       => 'The nominee photo must be a file',
                        'nominee_photo.*.extensions' => 'The nominee photo must be in jpg, png or jpeg format',
                        'nominee_photo.*.max'        => 'The nominee photo must less than 5 mb',

                        'nominee_nid_photo.*.required'   => 'The nominee nid photo filed is required',
                        'nominee_nid_photo.*.file'       => 'The nominee nid photo must be a file',
                        'nominee_nid_photo.*.extensions' => 'The nominee nid photo must be in jpg, png or jpeg format',
                        'nominee_nid_photo.*.max'        => 'The nominee nid photo must less than 5 mb',

                        'nominee_nid_back_photo.*.required'   => 'The nominee nid photo filed is required',
                        'nominee_nid_back_photo.*.file'       => 'The nominee nid photo must be a file',
                        'nominee_nid_back_photo.*.extensions' => 'The nominee nid photo must be in jpg, png or jpeg format',
                        'nominee_nid_back_photo.*.max'        => 'The nominee nid photo must less than 5 mb',

                        'nominee_signature.*.required'   => 'The nominee signature filed is required',
                        'nominee_signature.*.file'       => 'The nominee signature must be a file',
                        'nominee_signature.*.extensions' => 'The nominee signature must be in jpg, png or jpeg format',
                        'nominee_signature.*.max'        => 'The nominee signature photo must less than 5 mb',
                    ]
                );

                if ($validator->fails()) {
                    $errors = [];
                    foreach ($validator->errors()->messages() as $fieldName => $messages) {
                        $fieldNamePieces = explode('.', $fieldName);

                        if (count($fieldNamePieces) > 1) {
                            $errors[$fieldNamePieces[1]][$fieldNamePieces[0]] = $messages[0];
                        } else {
                            $errors[$fieldNamePieces[0]] = $messages[0];
                        }
                    }

                    ksort($errors);

                    return response()->json([
                        'status'  => 'error',
                        'message' => 'Form validation error',
                        'errors'  => $errors,
                    ], 400);
                }

                $investorImageName          = null;
                $investorNidImageName       = null;
                $investorNidBackImageName   = null;
                $investorSignatureImageName = null;
                $investorChequeImageName    = null;
                $investorLinkBoImageName    = null;

                $jointApplicantImageName        = null;
                $jointApplicantSignImageName    = null;
                $jointApplicantNidImageName     = null;
                $jointApplicantNidBackImageName = null;
                $jointApplicantBoImageName      = null;

                $nomineeImageName          = null;
                $nomineeNidImageName       = null;
                $nomineeNidBackImageName   = null;
                $nomineeSignatureImageName = null;
                $nomineeInfoId             = [];

                if ($request->hasFile('investor_photo')) {
                    $investorImage     = $request->file('investor_photo');
                    $investorImageName = 'investor_photo' . $request->investor_details_id . '_' . time() . '.' . $investorImage->getClientOriginalExtension();
                    $investorImage->move(public_path() . '/upload/investor_photo', $investorImageName);
                }

                if ($request->hasFile('investor_nid_photo')) {
                    $investorNidImage     = $request->file('investor_nid_photo');
                    $investorNidImageName = 'investor_nid_photo' . $request->investor_details_id . '_' . time() . '.' . $investorNidImage->getClientOriginalExtension();
                    $investorNidImage->move(public_path() . '/upload/investor_nid_photo', $investorNidImageName);
                }

                if ($request->hasFile('investor_nid_back_photo')) {
                    $investorNidImage     = $request->file('investor_nid_back_photo');
                    $investorNidBackImageName = 'investor_nid_back_photo' . $request->investor_details_id . '_' . time() . '.' . $investorNidImage->getClientOriginalExtension();
                    $investorNidImage->move(public_path() . '/upload/investor_nid_back_photo', $investorNidBackImageName);
                }

                if ($request->hasFile('investor_signature')) {
                    $investorSignatureImage     = $request->file('investor_signature');
                    $investorSignatureImageName = 'investor_signature' . $request->investor_details_id . '_' . time() . '.' . $investorSignatureImage->getClientOriginalExtension();
                    $investorSignatureImage->move(public_path() . '/upload/investor_signature', $investorSignatureImageName);
                }

                if ($request->hasFile('investor_cheque_photo')) {
                    $investorChequeImage     = $request->file('investor_cheque_photo');
                    $investorChequeImageName = 'investor_cheque_photo' . $request->investor_details_id . '_' . time() . '.' . $investorChequeImage->getClientOriginalExtension();
                    $investorChequeImage->move(public_path() . '/upload/investor_cheque_photo', $investorChequeImageName);
                }

                if ($request->hasFile('link_bo_photo')) {
                    $investorLinkBoImage     = $request->file('link_bo_photo');
                    $investorLinkBoImageName = 'link_bo_photo' . $request->investor_details_id . '_' . time() . '.' . $investorLinkBoImage->getClientOriginalExtension();
                    $investorLinkBoImage->move(public_path() . '/upload/link_bo_photo', $investorLinkBoImageName);
                }

                if ($request->hasFile('joint_applicant_photo')) {
                    $jointApplicantImage     = $request->file('joint_applicant_photo');
                    $jointApplicantImageName = 'joint_applicant_photo' . $request->investor_details_id . '_' . time() . '.' . $jointApplicantImage->getClientOriginalExtension();
                    $jointApplicantImage->move(public_path() . '/upload/joint_applicant_photo', $jointApplicantImageName);
                }

                if ($request->hasFile('joint_applicant_sign')) {
                    $jointApplicantSignImage     = $request->file('joint_applicant_sign');
                    $jointApplicantSignImageName = 'joint_applicant_sign' . $request->investor_details_id . '_' . time() . '.' . $jointApplicantSignImage->getClientOriginalExtension();
                    $jointApplicantSignImage->move(public_path() . '/upload/joint_applicant_sign', $jointApplicantSignImageName);
                }

                if ($request->hasFile('joint_applicant_nid')) {
                    $jointApplicantNidImage     = $request->file('joint_applicant_nid');
                    $jointApplicantNidImageName = 'joint_applicant_nid' . $request->investor_details_id . '_' . time() . '.' . $jointApplicantNidImage->getClientOriginalExtension();
                    $jointApplicantNidImage->move(public_path() . '/upload/joint_applicant_nid', $jointApplicantNidImageName);
                }

                if ($request->hasFile('joint_applicant_nid_back')) {
                    $jointApplicantNidImage     = $request->file('joint_applicant_nid_back');
                    $jointApplicantNidBackImageName = 'joint_applicant_nid_back' . $request->investor_details_id . '_' . time() . '.' . $jointApplicantNidImage->getClientOriginalExtension();
                    $jointApplicantNidImage->move(public_path() . '/upload/joint_applicant_nid_back', $jointApplicantNidBackImageName);
                }


                // if ($request->hasFile('joint_bo_photo')) {
                //     $jointApplicantBoImage     = $request->file('joint_bo_photo');
                //     $jointApplicantBoImageName = 'joint_bo_photo'.$request->investor_details_id.'_'.time().'.'.$jointApplicantBoImage->getClientOriginalExtension();
                //     $jointApplicantBoImage->move(public_path().'/upload/joint_bo_photo', $jointApplicantBoImageName);
                // }

                $nomineeImageFiles          = $request->file('nominee_photo');
                $nomineeNidImageFiles       = $request->file('nominee_nid_photo');
                $nomineeNidBackImageFiles   = $request->file('nominee_nid_back_photo');
                $nomineeSignatureImageFiles = $request->file('nominee_signature');

                for ($i = 0; $i <= max($fileCounterArr); $i++) {
                    if (!is_null($nomineeImageFiles[$i])) {
                        $nomineeImage     = $nomineeImageFiles[$i];
                        $nomineeImageName = 'nominee_photo_' . $i . '_' . $request->investor_details_id . '_' . time() . '.' . $nomineeImage->getClientOriginalExtension();
                        $nomineeImage->move(public_path() . '/upload/nominee_photo', $nomineeImageName);
                    }

                    if (!is_null($nomineeNidImageFiles[$i])) {
                        $nomineeNidImage     = $nomineeNidImageFiles[$i];
                        $nomineeNidImageName = 'nominee_nid_photo' . $request->investor_details_id . '_' . time() . '.' . $nomineeNidImage->getClientOriginalExtension();
                        $nomineeNidImage->move(public_path() . '/upload/nominee_nid_photo', $nomineeNidImageName);
                    }

                    if (!is_null($nomineeNidBackImageFiles[$i])) {
                        $nomineeNidImage     = $nomineeNidBackImageFiles[$i];
                        $nomineeNidBackImageName = 'nominee_nid_back_photo' . $request->investor_details_id . '_' . time() . '.' . $nomineeNidImage->getClientOriginalExtension();
                        $nomineeNidImage->move(public_path() . '/upload/nominee_nid_back_photo', $nomineeNidBackImageName);
                    }

                    if (!is_null($nomineeSignatureImageFiles[$i])) {
                        $nomineeSignatureImage     = $nomineeSignatureImageFiles[$i];
                        $nomineeSignatureImageName = 'nominee_signature' . $request->investor_details_id . '_' . time() . '.' . $nomineeSignatureImage->getClientOriginalExtension();
                        $nomineeSignatureImage->move(public_path() . '/upload/nominee_signature', $nomineeSignatureImageName);
                    }

                    $nomineeInfo = NomineeInfo::where('nominee_info_id', $request->nominee_info_id[$i])
                        ->where('investor_details_id', $request->investor_details_id)
                        ->first();

                    if (!is_null($nomineeInfo)) {
                        $nomineeInfoId[]                      = $nomineeInfo->nominee_info_id;
                        $nomineeInfo->nominee_applicant_photo = $nomineeImageName;
                        $nomineeInfo->nominee_nid             = $nomineeNidImageName;
                        $nomineeInfo->nominee_nid_back        = $nomineeNidBackImageName;
                        $nomineeInfo->nominee_applicant_sign  = $nomineeSignatureImageName;
                        $nomineeInfo->save();
                    }
                }

                if (is_null($investorDetail)) {
                    return response()->json([
                        'status'  => 'error',
                        'message' => 'Form validation error',
                    ], 400);
                }

                $investorDetail->applicant_photo = $investorImageName;
                $investorDetail->nid_copy        = $investorNidImageName;
                $investorDetail->nid_copy_back   = $investorNidBackImageName;
                $investorDetail->applicant_sign  = $investorSignatureImageName;
                $investorDetail->bank_copy       = $investorChequeImageName;
                $investorDetail->linked_bo_photo = $investorLinkBoImageName;
                $investorDetail->reg_page_no     = 3;
                $investorDetail->reg_is_complete = 'yes';
                $investorDetail->save();

                $jointApplicant = JointApplicant::where('investor_details_id', $investorDetail->investor_details_id)->first();

                if (!is_null($jointApplicant)) {
                    $jointApplicant->JOINT_APPLICANT_PHOTO = $jointApplicantImageName;
                    $jointApplicant->JOINT_APPLICANT_SIGN  = $jointApplicantSignImageName;
                    $jointApplicant->JOINT_BO_PHOTO        = $jointApplicantBoImageName;
                    $jointApplicant->NID_PHOTO             = $jointApplicantNidImageName;
                    $jointApplicant->NID_PHOTO_BACK        = $jointApplicantNidBackImageName;
                    $jointApplicant->save();
                }


                $recipient_email = $investorDetail->email;
                $mail_subject = "Welcome to Our Platform";
                $mail_message = "Welcome to our platform! We are excited to have you on board.";
        
                $this->sendWelcomeEmail($mail_subject,$recipient_email,$mail_message);



                return response()->json([
                    'status'              => 'success',
                    'message'             => 'Registration successful',
                    'investor_details_id' => $request->investor_details_id,
                    'nominee_info_id'     => $nomineeInfoId,
                ]);
            } else {
            }
        } catch (\Throwable $th) {
            return response()->json([
                'status'      => 'error',
                'message'     => 'Form submission error.',
                'dev_message' => $th->getMessage(),
                'file'        => $th->getFile(),
                'line'        => $th->getLine(),
            ], 400);
        }
    }

    public function changePassword(Request $request)
    {
        $validator = Validator::make(
            $request->all(),
            [
                'current_password'          => 'required',
                'new_password'              => 'required|string|min:8|confirmed',
                'new_password_confirmation' => 'required',
            ],
            [
                'current_password.required' => 'The current password field is required',
                'new_password.required'     => 'The new password field is required',
                'new_password.min'          => 'The new password field must be 8 char long',
                'new_password.confirmed'    => 'The new password must be match with confirmed password',
            ]
        );

        if ($validator->fails()) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Form validation error',
                'errors'  => $validator->errors()->messages(),
            ], 400);
        }

        auth()->user()->update([
            'password' => Hash::make($request->new_password)
        ]);

        return response()->json([
            'status'  => 'success',
            'message' => 'Password has been changed'
        ]);
    }

    public function signUp(Request $request, int $pageId)
    {
        try {
            $otpBypass    = true;
            $businessDate = BusinessDate::latest('business_date')->first();

            if ($pageId === 1) {
                /**
                 * Find the investor with email or mobile or portfolio code
                 */
                $validator = Validator::make(
                    $request->all(),
                    [
                        'email'          => 'required|email',
                        'contact_number' => 'required|digits:11',
                        'portfolio_code' => 'required',
                    ],
                    [
                        'email.required'          => 'The email field is required',
                        'email.email'             => 'The email field is not a valid email address',
                        'contact_number.required' => 'The contact number field is required',
                        'contact_number.digits'   => 'The contact number field must be 11 digits',
                        'portfolio_code.required' => 'The portfolio code field is required',
                    ]
                );

                if ($validator->fails()) {
                    return response()->json([
                        'status'  => 'error',
                        'message' => 'Form validation error',
                        'errors'  => $validator->errors()->messages(),
                    ], 400);
                } else {
                    $investorDetail = InvestorDetail::where('email', $request->email)
                        ->where('mobile', $request->contact_number)
                        ->where('portfolio_code', $request->portfolio_code)
                        ->first();

                    if (!is_null($investorDetail)) {
                        $user = User::where('email', $request->email)
                            ->orWhere('mobile_no', $request->contact_number)
                            ->orWhere('investor_code', $request->portfolio_code)
                            ->first();

                        if (is_null($user)) {

                            if ($otpBypass == false) {
                                $otpCode    = rand(123456, 999999);
                                $recordDate = now();
                                $message    = "Your OTP for signup is $otpCode. OTP will be valid for next 3 minutes";

                                $otp                      = new Otp();
                                $otp->message             = $message;
                                $otp->otp_code            = $otpCode;
                                $otp->mobile              = $request->contact_number;
                                $otp->otp_valid_date_time = now()->addMinutes(4);
                                $otp->business_date       = $businessDate->business_date;
                                $otp->record_date         = $recordDate;

                                if ($otp->save()) {
                                    $smsHelper = new SmsHelper();
                                    $smsResponse = json_decode($smsHelper->sendSingleSms($investorDetail->mobile, $message)->content());
                                    // $smsResponse = json_decode($smsHelper->sendSingleSms('01774911240', $message)->content());

                                    if ($smsResponse->status == 'error') {
                                        $otp->delete();

                                        return response()->json([
                                            'status'  => 'error',
                                            'message' => 'We are unable to send OTP for veryfying your request',
                                        ], 400);
                                    }

                                    $otp->msg_id = $smsResponse->message_info->csms_id;
                                    $otp->save();

                                    return response()->json([
                                        'status'  => 'success',
                                        'message' => 'Form submitted successfully',
                                    ]);
                                } else {
                                    return response()->json([
                                        'status'  => 'error',
                                        'message' => 'We are unable to send OTP for veryfying your request',
                                    ], 400);
                                }
                            } else {
                                return response()->json([
                                    'status'  => 'success',
                                    'message' => 'Form submitted successfully',
                                ]);
                            }
                        }

                        return response()->json([
                            'status'  => 'error',
                            'message' => 'An investor already registered with providing details',
                        ], 400);
                    }

                    return response()->json([
                        'status'  => 'error',
                        'message' => 'We could not found you details in our system with providing details',
                    ], 400);
                }
            } elseif ($pageId === 2) {
                /**
                 * verify otp
                 */
                $validator = Validator::make(
                    $request->all(),
                    [
                        'otp_code' => 'required|numeric|digits:6',
                    ],
                    [
                        'otp_code.required' => 'The otp field is required',
                        'otp_code.numeric'  => 'The otp field must be a number',
                        'otp_code.digits'   => 'The otp field must be 6 digits',
                    ]
                );

                if ($validator->fails()) {
                    return response()->json([
                        'status'  => 'error',
                        'message' => 'Form validation error',
                        'errors'  => $validator->errors()->messages(),
                    ], 400);
                }

                $otp = Otp::where('otp_code', $request->otp_code)
                    ->where('otp_verify', null)
                    ->where('otp_valid_date_time', '>=', now())
                    ->first();

                if ($otpBypass == false) {
                    if (is_null($otp)) {
                        return response()->json([
                            'status'     => 'error',
                            'otp_verify' => false,
                        ], 400);
                    }

                    $otp->otp_verify = 'yes';
                    $otp->save();
                }

                return response()->json([
                    'status'     => 'success',
                    'otp_verify' => true,
                ]);
            } elseif ($pageId === 3) {
                /**
                 * Insert/update investor details
                 */
                $validator = Validator::make(
                    $request->all(),
                    [
                        'email'                 => 'required|email',
                        'contact_number'        => 'required|digits:11',
                        'portfolio_code'        => 'required',
                        'password'              => 'required|digits:8|confirmed',
                        'password_confirmation' => 'required',
                    ],
                    [
                        'email.required'                 => 'The email field is required',
                        'email.email'                    => 'The email field is not a valid email address',
                        'contact_number.required'        => 'The contact number field is required',
                        'contact_number.digits'          => 'The contact number field must be 11 digits',
                        'portfolio_code.required'        => 'The portfolio code field is required',
                        'password.required'              => 'The password field is required',
                        'password.digits'                => 'The password field must be 8 digits',
                        'password.confirmed'             => 'The password field must be matched with confirm password field',
                        'password_confirmation.required' => 'The confirm password field is required',
                    ]
                );

                // dd($validator->fails());

                if ($validator->fails()) {
                    return response()->json([
                        'status'  => 'error',
                        'message' => 'Form validation error',
                        'errors'  => $validator->errors()->messages(),
                    ], 400);
                }

                $investorDetail = InvestorDetail::where('portfolio_code', $request->portfolio_code)->first();

                $user                = new User();
                $user->name          = $investorDetail->investor_name;
                $user->investor_code = $request->portfolio_code;
                $user->email         = $request->email;
                $user->password      = Hash::make($request->password);
                $user->mobile_no     = $request->contact_number;
                $user->created_at    = now();
                $user->updated_at    = now();
                $user->save();

                return response()->json([
                    'status'  => 'success',
                    'message' => 'Sign up successfull',
                ]);
            } else {
                return response()->json([
                    'status'  => 'success',
                    'message' => 'Unknown request',
                ], 400);
            }
        } catch (\Throwable $th) {
            return response()->json([
                'status'      => 'error',
                'dev_message' => $th->getMessage(),
                'file'        => $th->getFile(),
                'line'        => $th->getLine(),
                'message'     => 'Form submit unsuccessful',
            ], 400);
        }
    }

    public function forgotPassword(Request $request, int $pageId)
    {
        try {
            $otpBypass = true;
            $businessDate = BusinessDate::latest('business_date')->first();

            if ($pageId === 1) {
                $validator = Validator::make(
                    $request->all(),
                    [
                        'email_or_mobile' => 'required',
                    ],
                    [
                        'email_or_mobile.required' => 'The email or mobile field is required',
                    ]
                );

                if ($validator->fails()) {
                    return response()->json([
                        'status'  => 'error',
                        'message' => 'Form validation error',
                        'errors'  => $validator->errors()->messages(),
                    ], 400);
                }

                $user = User::where('email', $request->email_or_mobile)
                    ->orWhere('mobile_no', $request->email_or_mobile)
                    ->exclude(['password'])
                    ->first();

                if (is_null($user)) {
                    return response()->json([
                        'status'  => 'error',
                        'message' => 'We cound not found any investor with providing email or mobile'
                    ], 400);
                }

                if ($otpBypass == false) {
                    $otpCode    = rand(123456, 999999);
                    $recordDate = now();
                    $message    = "Your OTP for recover your password is $otpCode. OTP will be valid for next 3 minutes";

                    $otp                      = new Otp();
                    $otp->message             = $message;
                    $otp->otp_code            = $otpCode;
                    $otp->mobile              = $request->contact_number;
                    $otp->otp_valid_date_time = now()->addMinutes(4);
                    $otp->business_date       = $businessDate->business_date;
                    $otp->record_date         = $recordDate;

                    if ($otp->save()) {
                        $smsHelper = new SmsHelper();
                        $smsResponse = json_decode($smsHelper->sendSingleSms($user->mobile_no, $message)->content());
                        // $smsResponse = json_decode($smsHelper->sendSingleSms('01774911240', $message)->content());

                        if ($smsResponse->status == 'error') {
                            $otp->delete();

                            return response()->json([
                                'status'  => 'error',
                                'message' => 'We are unable to send OTP for veryfying your request',
                            ], 400);
                        }

                        $otp->msg_id = $smsResponse->message_info->csms_id;
                        $otp->save();
                    } else {
                        return response()->json([
                            'status'  => 'error',
                            'message' => 'We are unable to send OTP for veryfying your request',
                        ], 400);
                    }
                }

                return response()->json([
                    'status'  => 'success',
                    'user' => $user,
                    'message' => 'OTP sent to your mobile successfully',
                ]);
            }

            if ($pageId === 2) {
                /**
                 * verify otp
                 */
                $validator = Validator::make(
                    $request->all(),
                    [
                        'otp_code' => 'required|numeric|digits:6',
                    ],
                    [
                        'otp_code.required' => 'The otp field is required',
                        'otp_code.numeric'  => 'The otp field must be a number',
                        'otp_code.digits'   => 'The otp field must be 6 digits',
                    ]
                );

                if ($validator->fails()) {
                    return response()->json([
                        'status'  => 'error',
                        'message' => 'Form validation error',
                        'errors'  => $validator->errors()->messages(),
                    ], 400);
                }

                $otp = Otp::where('otp_code', $request->otp_code)
                    ->where('otp_verify', null)
                    ->where('otp_valid_date_time', '>=', now())
                    ->first();

                if ($otpBypass == false) {
                    if (is_null($otp)) {
                        return response()->json([
                            'status'     => 'error',
                            'otp_verify' => false,
                            'message' => 'OTP verification unsuccessful'
                        ], 400);
                    }

                    $otp->otp_verify = 'yes';
                    $otp->save();
                }

                return response()->json([
                    'status'     => 'success',
                    'otp_verify' => true,
                    'message' => 'OTP verification successful'
                ]);
            }

            if ($pageId === 3) {
                /**
                 * change password
                 */
                $validator = Validator::make(
                    $request->all(),
                    [
                        'user_id'                   => 'required|exists:users,id',
                        'new_password'              => 'required|string|min:8|confirmed',
                        'new_password_confirmation' => 'required',
                    ],
                    [
                        'user_id.required'       => 'The user id is missing',
                        'user_id.exists'         => 'The user id is invalid',
                        'new_password.required'  => 'The new password field is required',
                        'new_password.min'       => 'The new password field must be 8 char long',
                        'new_password.confirmed' => 'The new password must be match with confirmed password',
                    ]
                );

                if ($validator->fails()) {
                    return response()->json([
                        'status'  => 'error',
                        'message' => 'Form validation error',
                        'errors'  => $validator->errors()->messages(),
                    ], 400);
                }

                User::find($request->user_id)->update([
                    'password' => Hash::make($request->new_password)
                ]);

                return response()->json([
                    'status'  => 'success',
                    'message' => 'Password has been changed'
                ]);
            }
        } catch (\Throwable $th) {
            return response()->json([
                'status'      => 'error',
                'dev_message' => $th->getMessage(),
                'message'     => 'Form validation error',
            ], 400);
        }
    }

    public function operationMode()
    {
        $operationMode = [
            '1' => 'Single',
            '2' => 'Joint',
        ];

        return response()->json([
            'status'         => 'success',
            'operation_mode' => $operationMode,
        ]);
    }

    public function sendWelcomeEmail($mail_subject,$recipient_email,$mail_message)
    {
        // for blade template
        // Mail::send('emails.welcome', [], function ($message) use ($recipientEmail, $mail_subject) {
        //     $message->to($recipientEmail)
        //             ->subject($mail_subject);
        // });

        return true;
        
        Mail::raw($mail_message, function ($message) use ($recipient_email, $mail_subject) {
            $message->to($recipient_email)
                    ->subject($mail_subject);
        });

        return true;

    }
}
