<?php

namespace App\Http\Controllers\Investor;

use Carbon\Carbon;
use App\Models\BusinessDate;
use Illuminate\Http\Request;
use App\Models\InvestorDetail;
use App\Http\Controllers\Controller;
use App\Models\InvestorChangeRequest;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class InvestorInformationController extends Controller
{
    public function personalInformation()
    {
        $user = Auth::user();
        $investorDetail = InvestorDetail::where('portfolio_code', $user->investor_code)->with(['templateDetails', 'branchDetails'],)->first();

        // dd($investorDetail->branchDetails->bankDetails);
        
        
        return response()->json([
            'status'   => 'success',
            'investor' => [
                'investor_name'     => $investorDetail->investor_name,
                'father_name'       => $investorDetail->father_name,
                'mother_name'       => $investorDetail->mother_name,
                'residency'         => $investorDetail->residency,
                'portfolio_code'    => $investorDetail->portfolio_code,
                'mailing_address'   => $investorDetail->mailing_address,
                'mobile'            => $investorDetail->mobile,
                'email'             => $investorDetail->email,
                'nid'               => $investorDetail->nid,
                'bank_account_no'   => $investorDetail->bank_account_no,
                'bo_account_no'     => $investorDetail->bo_account_no,
                'bank_id'           => $investorDetail->branchDetails->bankDetails->org_id,
                'bank_name'         => $investorDetail->branchDetails->bankDetails->org_name,
                'branch_id'         => $investorDetail->branchDetails->org_branch_id,
                'branch_name'       => $investorDetail->branchDetails->branch_name,
                'branch_routing_no' => $investorDetail->branchDetails->routing_no,
                'product_name'      => $investorDetail->templateDetails->product_name,
                'dob'               => empty($investorDetail->dob) || is_null($investorDetail->dob) ? 'N/A' : Carbon::createFromFormat( 'Y-m-d H:i:s', $investorDetail->dob)->format('d-M-Y'),
            ]
        ]);
    }

    public function submitChangeRequest(Request $request)
    {
        $validator = Validator::make(
            $request->all(),
            [
                'name'            => 'required',
                'mailing_address' => 'required',
                'email_address'   => 'required',
                'mobile_no'       => 'required',
                'bank_id'         => 'required',
                'branch_id'       => 'required',
                'bank_account_no' => 'required',
                'bo_account_no'   => 'required',
                'bank_leaf'       => 'file|extensions:jpg,png,jpeg,pdf',
            ],
            [
                'name.required'            => 'The name filed is required',
                'mailing_address.required' => 'The mailing address filed is required',
                'email_address.required'   => 'The email address filed is required',
                'mobile_no.required'       => 'The mobile number filed is required',
                'bank_id.required'         => 'The bank filed is required',
                'branch_id.required'       => 'The branch filed is required',
                'bank_account_no.required' => 'The bank account number filed is required',
                'bo_account_no.required'   => 'The bo account number filed is required',
                'bank_leaf.file'           => 'The bank leaf must be a file',
                'bank_leaf.extensions'     => 'The bank leaf must be in jpg, png, jpeg or pdf format',
            ]
        );

        if ($validator->fails()) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Form validation error',
                'errors' => $validator->errors()->messages(),
            ], 400);
        } else {
            try {
                $bankLeadFileName = null;
                $user                 = Auth::user();
                $businessDate         = BusinessDate::latest('business_date')->first();

                if ($request->hasFile('bank_leaf')) {
                    $bankLeadFile     = $request->file('bank_leaf');
                    $bankLeadFileName = 'bank_leaf_'.$user->investor_code.'_'.time().'.'.$bankLeadFile->getClientOriginalExtension();
                    $bankLeadFile->move(public_path().'/upload/bank_leaf', $bankLeadFileName);    
                }


                $investorChangeRequest                      = new InvestorChangeRequest();
                $investorChangeRequest->investor_details_id = $user->investorDetail->investor_details_id;
                $investorChangeRequest->name                = $request->name;
                $investorChangeRequest->mailing_address     = $request->mailing_address;
                $investorChangeRequest->email               = $request->email;
                $investorChangeRequest->mobile              = $request->mobile_no;
                $investorChangeRequest->bank_id             = $request->bank_id;
                $investorChangeRequest->branch_id           = $request->branch_id;
                $investorChangeRequest->account_no          = $request->bank_account_no;
                $investorChangeRequest->bo_account_no       = $request->bo_account_no;
                $investorChangeRequest->bank_leaf           = $bankLeadFileName;
                $investorChangeRequest->business_date       = $businessDate->business_date;
                $investorChangeRequest->record_date         = now();

                if ($investorChangeRequest->save()) {
                    return response()->json([
                        'status'      => 'success',
                        'message'   => 'Change request submitted successfully',
                    ]);
                }
            } catch (\Throwable $th) {
                return response()->json([
                    'status'      => 'error',
                    'dev_message' => $th->getMessage(),
                    'message'      => 'Change request form submit unsuccessful',
                ], 400);
            }
        }
    }
}
