<?php

namespace App\Http\Controllers\Report;

use Carbon\Carbon;
use App\Models\BusinessDate;
use Illuminate\Http\Request;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class PortfolioStatementController extends Controller
{
    public function portfolioStatementDownload(Request $request)
    {
		$validator = Validator::make(
            $request->all(),
            [
                'as_on_date'  => 'required|date_format:Y-m-d',
            ],
            [
                'as_on_date.required'          => 'The as on date filed is required',
                'as_on_date.date_format'       => 'The as on date field must be in Y-m-d format',
            ]
        );

        if ($validator->fails()) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Form validation error',
                'errors' => $validator->errors()->messages(),
            ], 400);
        }

        $data = [
            'title' => 'Portfolio Statement (Date Wise)',
        ];

		$user                    = Auth::user();
		$investorDetail          = $user->investorDetail;
		$businessDate            = BusinessDate::latest('business_date')->first();
		$compareDate             = Carbon::createFromFormat('Y-m-d', '2022-12-31');
		$requestDate             = Carbon::createFromFormat('Y-m-d', $request->as_on_date);
		$data['account_type']    = $investorDetail->templateDetails->product_short_name;
		$data['account_no']      = $investorDetail->portfolio_code;
		$data['bo_no']           = $investorDetail->bo_account_no;
		$data['account_date']    = Carbon::createFromFormat('Y-m-d H:i:s', $investorDetail->business_date)->format(env('OUTPUT_DATE_FORMAT'));
		$data['account_name']    = $investorDetail->investor_name;
		$data['account_address'] = $investorDetail->mailing_address;
		$data['report_date']     = Carbon::createFromFormat('Y-m-d', $requestDate->format('Y-m-d'))->format(env('OUTPUT_DATE_FORMAT'));

		if ($investorDetail->account_enable_disable == 'e') {
			$data['account_status'] = 'Active';
		}

		if ($investorDetail->account_close == 'y') {
			$data['account_status'] = 'Close';
		}

		if ($investorDetail->account_suspend == 'y') {
			$data['account_status'] = 'Suspend';
		}


        if ($requestDate->gte($compareDate)) {
            $portfolioStatementSql = "
            
                SELECT LTEMP.PRODUCT_SHORT_NAME AS PRODUCT_NAME,
						 COALESCE (INDV.INSTITUTION_NAME, INDV.INVESTOR_NAME)							 AS INVESTOR_NAME,
						 COALESCE (INDV.MAILING_ADDRESS, INDV.PERMANENT_ADDRESS)						 AS INVESTOR_ADDRESS,
						 INDV.PORTFOLIO_CODE							AS PORTFOLIO_CODE,
						 INDV.BO_ACCOUNT_NO								AS BO_ACCOUNT_NO,
						 INDV.MOBILE									AS MOBILE,
						 INDV.TEMPLATE_ID								AS TEMPLATE_ID,
						 INDV.TENURE_YEAR								AS TENURE_YEAR,
						 INDV.TENURE_MATURE_DATE					 	AS TENURE_MATURE_DATE,
						 INDV.MONTHLY_SCHEME						 	AS MONTHLY_SCHEME,
						 INDV.BUSINESS_DATE							 	AS DATE_OF_ACC_OPENING,
						 INDV.ACCOUNT_CLOSE							 	AS ACCOUNT_CLOSE,
						 INDV.ACCOUNT_SUSPEND							AS ACCOUNT_SUSPEND,
						 INDV.ACCOUNT_ENABLE_DISABLE				 	AS ACCOUNT_ENABLE_DISABLE,
						 (CASE
							  WHEN INDV.IS_MARGIN_LOAN_ACTIVE = 'y' THEN 'Margin'
							  ELSE 'Non Margin'
						  END)							 		AS MARGIN_NONMARGIN,
						 STOCK_INFO.INSTRUMENT_DETAILS_ID			 	AS INSTRUMENT_DETAILS_ID,
						 STOCK_INFO.TOTAL_SHARE + NVL (FRACTION_INFO.FRACTION_STOCK_DIV, 0)			 AS TOTAL_SHARE,
						 STOCK_INFO.AVG_RATE							 AS AVG_RATE,
						   STOCK_INFO.TOTAL_COST_AMOUNT
						 + (STOCK_INFO.AVG_RATE * NVL (FRACTION_INFO.FRACTION_STOCK_DIV, 0))
							 AS TOTAL_COST_AMOUNT,
						   STOCK_INFO.TOTAL_MKT_AMOUNT
						 + +(  NVL (STOCK_INFO.CLOSE_PRICE, 0.1)
							 * NVL (FRACTION_INFO.FRACTION_STOCK_DIV, 0))
							 AS TOTAL_MKT_AMOUNT,
						 STOCK_INFO.MATURED_SHARE					 	AS MATURED_SHARE,
						 STOCK_INFO.INSTRUMENT_NAME					 	AS INSTRUMENT_NAME,
						 DECODE(STOCK_INFO.SECTOR_ID,83,STOCK_INFO.AVG_RATE,NVL (STOCK_INFO.CLOSE_PRICE, 0.1))AS CLOSE_PRICE,
						--   NVL (STOCK_INFO.CLOSE_PRICE, 0.1)							 AS CLOSE_PRICE,
						 STOCK_INFO.CHARGE							 	AS CHARGE,
						 STOCK_INFO.SECTOR_NAME							AS SECTOR_NAME,
						 STOCK_INFO.SYMBOL							 	AS SYMBOL,
						 STOCK_INFO.MARGIN_OR_NONMARGIN_FLAG		 	AS MARGIN_OR_NONMARGIN_FLAG,
						 STOCK_INFO.YCP							 		AS YCP,
						 STOCK_INFO.RATIO_VALUE							AS RATIO_VALUE
					FROM IS_INVESTOR_DETAILS INDV
						 LEFT JOIN
						 (SELECT DISTINCT ISCMS.PORTFOLIO_CODE,
								 ISCMS.INSTRUMENT_DETAILS_ID
									 AS INSTRUMENT_DETAILS_ID,
								 ISCMS.TOTAL_SHARE									AS TOTAL_SHARE,
								 ISCMS.NEW_AVERAGE_RATE								AS AVG_RATE,
								 NVL (ISCMS.TOTAL_SHARE * ISCMS.NEW_AVERAGE_RATE, 0)
									 AS TOTAL_COST_AMOUNT,
								 NVL (P_CAP_MKT_PRICE.CLOSE_PRICE * ISCMS.TOTAL_SHARE, 0)
									 AS TOTAL_MKT_AMOUNT,
								 ISCMS.MATURED_SHARE					AS MATURED_SHARE,
								 LSID.INSTRUMENT_NAME					AS INSTRUMENT_NAME,
								 LSID.SECTOR_ID					 		AS SECTOR_ID,
								 P_CAP_MKT_PRICE.CLOSE_PRICE
									 AS CLOSE_PRICE,
								 0
									 AS CHARGE,
								 ''
									 AS SECTOR_NAME,
								 MWINSTDET.SYMBOL
									 AS SYMBOL,
								 MWINSTDET.MARGIN_OR_NONMARGIN_FLAG,
								 0
									 AS YCP,
								 PE_RATIO.RATIO_VALUE
									 AS RATIO_VALUE
							FROM IS_CAP_MKT_STOCK     ISCMS,
								 (SELECT SYMBOL,
										 INSTRUMENT_DETAILS_ID,
										 MARGIN_OR_NONMARGIN_FLAG
									FROM LS_MARKET_WISE_INSTRUMENT
								   WHERE MARKET_DETAILS_ID = 61) MWINSTDET,
								 LS_INSTRUMENT_DETAILS LSID
								 LEFT JOIN
								 (SELECT P_PE_RATIO.INSTRUMENT_DETAILS_ID,
										 P_PE_RATIO.RATIO_VALUE
									FROM P_PE_RATIO
								   WHERE P_PE_RATIO.BUSINESS_DATE =
										 (SELECT MAX (P_PE_RATIO.BUSINESS_DATE)
											FROM P_PE_RATIO
										   WHERE P_PE_RATIO.BUSINESS_DATE <= :business_date)) PE_RATIO
									 ON PE_RATIO.INSTRUMENT_DETAILS_ID =
										LSID.INSTRUMENT_DETAILS_ID
								 LEFT JOIN
								 (SELECT CMPH.INSTRUMENT_DETAILS_ID, CMPH.CLOSE_PRICE
									FROM P_CAP_MKT_PRICE_HISTORY CMPH,
										 LS_INSTRUMENT_DETAILS  INSTDET
								   WHERE     INSTDET.INSTRUMENT_DETAILS_ID =
											 CMPH.INSTRUMENT_DETAILS_ID
										 AND CMPH.BUSINESS_DATE =
											 (SELECT MAX (
														 P_CAP_MKT_PRICE_HISTORY.BUSINESS_DATE)
												FROM P_CAP_MKT_PRICE_HISTORY
											   WHERE     P_CAP_MKT_PRICE_HISTORY.BUSINESS_DATE <= :business_date
													 AND INSTDET.INSTRUMENT_DETAILS_ID =
														 P_CAP_MKT_PRICE_HISTORY.INSTRUMENT_DETAILS_ID))
								 P_CAP_MKT_PRICE
									 ON P_CAP_MKT_PRICE.INSTRUMENT_DETAILS_ID =
										LSID.INSTRUMENT_DETAILS_ID
						   WHERE     ISCMS.CAP_MKT_STOCK_ID =
									 (SELECT MAX (ISCMSD.CAP_MKT_STOCK_ID)
										FROM IS_CAP_MKT_STOCK ISCMSD
									   WHERE     ISCMSD.INSTRUMENT_DETAILS_ID =
												 ISCMS.INSTRUMENT_DETAILS_ID
											 AND ISCMS.PORTFOLIO_CODE = ISCMSD.PORTFOLIO_CODE
											 AND LSID.INSTRUMENT_DETAILS_ID =
												 ISCMSD.INSTRUMENT_DETAILS_ID
											 AND ISCMSD.BUSINESS_DATE <= :business_date)
								 AND ISCMS.PORTFOLIO_CODE = :investor_code
								 AND MWINSTDET.INSTRUMENT_DETAILS_ID =
									 LSID.INSTRUMENT_DETAILS_ID) STOCK_INFO
							 ON STOCK_INFO.PORTFOLIO_CODE = INDV.PORTFOLIO_CODE
						 LEFT JOIN
						 ( 			  
						SELECT FRACTION_DETAILS.PORTFOLIO_CODE             AS PORTFOLIO_CODE,
							 FRACTION_DETAILS.INSTRUMENT_DETAILS_ID        AS INSTRUMENT_DETAILS_ID,
							 SUM (FRACTION_DETAILS.FRACTION_STOCK_DIV)     AS FRACTION_STOCK_DIV
						FROM (  SELECT INVDET.PORTFOLIO_CODE
										   AS PORTFOLIO_CODE,
									   INSTDET.INSTRUMENT_DETAILS_ID
										   AS INSTRUMENT_DETAILS_ID,
									   NVL (SUM (STOCKDIV.STOCK_DIV_FRACTION - STOCKDIV.STOCK_DIV),
											0)
										   AS FRACTION_STOCK_DIV
								  FROM IS_INVESTOR_DETAILS INVDET,
									   IS_STOCK_DIVIDEND  STOCKDIV,
									   LS_INSTRUMENT_DETAILS INSTDET
								 WHERE     INVDET.PORTFOLIO_CODE = STOCKDIV.PORTFOLIO_CODE
									   AND INSTDET.INSTRUMENT_DETAILS_ID =
										   STOCKDIV.INSTRUMENT_DETAILS_ID
									   AND INVDET.PORTFOLIO_CODE = :investor_code
									   AND STOCKDIV.BUSINESS_DATE <= :business_date
									   AND (STOCKDIV.STOCK_DIV_FRACTION - STOCKDIV.STOCK_DIV) > 0
									   AND STOCKDIV.FRACTION_SHARE_RECEIVED = 'n'
							  GROUP BY INVDET.PORTFOLIO_CODE, INSTDET.INSTRUMENT_DETAILS_ID
							  UNION ALL
								SELECT INVDET.PORTFOLIO_CODE
										   AS PORTFOLIO_CODE,
									   INSTDET.INSTRUMENT_DETAILS_ID
										   AS INSTRUMENT_DETAILS_ID,
									   NVL (SUM (STOCKDIV.STOCK_DIV_FRACTION - STOCKDIV.STOCK_DIV),
											0)
										   AS FRACTION_STOCK_DIV
								  FROM IS_INVESTOR_DETAILS INVDET,
									   IS_STOCK_DIVIDEND  STOCKDIV,
									   LS_INSTRUMENT_DETAILS INSTDET
								 WHERE     INVDET.PORTFOLIO_CODE = STOCKDIV.PORTFOLIO_CODE
									   AND INSTDET.INSTRUMENT_DETAILS_ID =
										   STOCKDIV.INSTRUMENT_DETAILS_ID
									   AND INVDET.PORTFOLIO_CODE = :investor_code
									   AND (STOCKDIV.STOCK_DIV_FRACTION - STOCKDIV.STOCK_DIV) > 0
									   AND STOCKDIV.FRACTION_SHARE_RECEIVED = 'y'
									   AND STOCKDIV.BUSINESS_DATE >= '31-Dec-2022'
									   AND STOCKDIV.BUSINESS_DATE <= :business_date
									   AND STOCKDIV.FRACTION_SHARE_RECEIVED_DATE >  :business_date
							  GROUP BY INVDET.PORTFOLIO_CODE, INSTDET.INSTRUMENT_DETAILS_ID)
							 FRACTION_DETAILS
						GROUP BY FRACTION_DETAILS.PORTFOLIO_CODE,
							 FRACTION_DETAILS.INSTRUMENT_DETAILS_ID	
						  )
						 FRACTION_INFO
							 ON FRACTION_INFO.INSTRUMENT_DETAILS_ID =
								STOCK_INFO.INSTRUMENT_DETAILS_ID
						 LEFT JOIN LS_TEMPLATE LTEMP ON LTEMP.TEMPLATE_ID = INDV.TEMPLATE_ID
				   WHERE INDV.PORTFOLIO_CODE = :investor_code
				ORDER BY INDV.PORTFOLIO_CODE, MARGIN_OR_NONMARGIN_FLAG DESC, SYMBOL ASC
            
            ";
        } else {
            $portfolioStatementSql = "
                SELECT LTEMP.PRODUCT_SHORT_NAME AS PRODUCT_NAME,
							COALESCE (INDV.INSTITUTION_NAME, INDV.INVESTOR_NAME)		AS INVESTOR_NAME,
							COALESCE (INDV.MAILING_ADDRESS, INDV.PERMANENT_ADDRESS) 	AS INVESTOR_ADDRESS,
						INDV.PORTFOLIO_CODE					AS PORTFOLIO_CODE,
						INDV.BO_ACCOUNT_NO					AS BO_ACCOUNT_NO,
						INDV.MOBILE							AS MOBILE,
						INDV.TEMPLATE_ID					AS TEMPLATE_ID,
						INDV.TENURE_YEAR					AS TENURE_YEAR,
						INDV.TENURE_MATURE_DATE				AS TENURE_MATURE_DATE,
						INDV.MONTHLY_SCHEME					AS MONTHLY_SCHEME,
						INDV.BUSINESS_DATE					AS DATE_OF_ACC_OPENING,
						INDV.ACCOUNT_CLOSE					AS ACCOUNT_CLOSE,
						INDV.ACCOUNT_SUSPEND				AS ACCOUNT_SUSPEND,
						INDV.ACCOUNT_ENABLE_DISABLE			AS ACCOUNT_ENABLE_DISABLE,
						(CASE
								WHEN INDV.IS_MARGIN_LOAN_ACTIVE = 'y' THEN 'Margin'
								ELSE 'Non Margin'
							END)							AS MARGIN_NONMARGIN,
						STOCK_INFO.INSTRUMENT_DETAILS_ID	AS INSTRUMENT_DETAILS_ID,
						STOCK_INFO.TOTAL_SHARE				AS TOTAL_SHARE,
						STOCK_INFO.AVG_RATE					AS AVG_RATE,
						STOCK_INFO.TOTAL_COST_AMOUNT		AS TOTAL_COST_AMOUNT,
						STOCK_INFO.TOTAL_MKT_AMOUNT			AS TOTAL_MKT_AMOUNT,
						STOCK_INFO.MATURED_SHARE			AS MATURED_SHARE,
						STOCK_INFO.INSTRUMENT_NAME			AS INSTRUMENT_NAME,
						NVL (STOCK_INFO.CLOSE_PRICE, 0.1)	AS CLOSE_PRICE,
						STOCK_INFO.CHARGE					AS CHARGE,
						STOCK_INFO.SECTOR_NAME				AS SECTOR_NAME,
						STOCK_INFO.SYMBOL					AS SYMBOL,
						STOCK_INFO.MARGIN_OR_NONMARGIN_FLAG					AS MARGIN_OR_NONMARGIN_FLAG,
						STOCK_INFO.YCP						AS YCP,
						STOCK_INFO.RATIO_VALUE				AS RATIO_VALUE
					FROM IS_INVESTOR_DETAILS  INDV
						LEFT JOIN
						(SELECT ISCMS.PORTFOLIO_CODE,
								ISCMS.INSTRUMENT_DETAILS_ID
									AS INSTRUMENT_DETAILS_ID,
								ISCMS.FRACTION_TOTAL_SHARE
									AS TOTAL_SHARE,
								ISCMS.NEW_AVERAGE_RATE
									AS AVG_RATE,
								NVL (ISCMS.FRACTION_TOTAL_SHARE * ISCMS.NEW_AVERAGE_RATE, 0)
									AS TOTAL_COST_AMOUNT,
								NVL (P_CAP_MKT_PRICE.CLOSE_PRICE * ISCMS.FRACTION_TOTAL_SHARE, 0)
									AS TOTAL_MKT_AMOUNT,
								ISCMS.MATURED_SHARE
									AS MATURED_SHARE,
								LSID.INSTRUMENT_NAME
									AS INSTRUMENT_NAME,
								P_CAP_MKT_PRICE.CLOSE_PRICE
									AS CLOSE_PRICE,
								0
									AS CHARGE,
								''
									AS SECTOR_NAME,
								MWINSTDET.SYMBOL
									AS SYMBOL,
								MWINSTDET.MARGIN_OR_NONMARGIN_FLAG,
								0
									AS YCP,
								PE_RATIO.RATIO_VALUE
									AS RATIO_VALUE
							FROM IS_CAP_MKT_STOCK       ISCMS,
								(SELECT SYMBOL,
										INSTRUMENT_DETAILS_ID,
										MARGIN_OR_NONMARGIN_FLAG
									FROM LS_MARKET_WISE_INSTRUMENT
									WHERE MARKET_DETAILS_ID = 61) MWINSTDET,
								LS_INSTRUMENT_DETAILS  LSID
								LEFT JOIN
								(SELECT P_PE_RATIO.INSTRUMENT_DETAILS_ID,
										P_PE_RATIO.RATIO_VALUE
									FROM P_PE_RATIO
									WHERE P_PE_RATIO.BUSINESS_DATE =
										(SELECT MAX (P_PE_RATIO.BUSINESS_DATE)
											FROM P_PE_RATIO
											WHERE P_PE_RATIO.BUSINESS_DATE <= :business_date))
								PE_RATIO
									ON PE_RATIO.INSTRUMENT_DETAILS_ID =
										LSID.INSTRUMENT_DETAILS_ID
								LEFT JOIN
								(SELECT CMPH.INSTRUMENT_DETAILS_ID, CMPH.CLOSE_PRICE
									FROM P_CAP_MKT_PRICE_HISTORY CMPH, LS_INSTRUMENT_DETAILS INSTDET
									WHERE     INSTDET.INSTRUMENT_DETAILS_ID = CMPH.INSTRUMENT_DETAILS_ID
										AND CMPH.BUSINESS_DATE =
											(SELECT MAX (P_CAP_MKT_PRICE_HISTORY.BUSINESS_DATE)
												FROM P_CAP_MKT_PRICE_HISTORY
												WHERE     P_CAP_MKT_PRICE_HISTORY.BUSINESS_DATE <=  :business_date
													AND INSTDET.INSTRUMENT_DETAILS_ID =
														P_CAP_MKT_PRICE_HISTORY.INSTRUMENT_DETAILS_ID)
											)
								P_CAP_MKT_PRICE
									ON P_CAP_MKT_PRICE.INSTRUMENT_DETAILS_ID =
										LSID.INSTRUMENT_DETAILS_ID
							WHERE     ISCMS.CAP_MKT_STOCK_ID =
									(SELECT MAX (ISCMSD.CAP_MKT_STOCK_ID)
										FROM IS_CAP_MKT_STOCK ISCMSD
										WHERE     ISCMSD.INSTRUMENT_DETAILS_ID =
												ISCMS.INSTRUMENT_DETAILS_ID
											AND ISCMS.PORTFOLIO_CODE = ISCMSD.PORTFOLIO_CODE
											AND LSID.INSTRUMENT_DETAILS_ID =
												ISCMSD.INSTRUMENT_DETAILS_ID
											AND ISCMSD.BUSINESS_DATE <= :business_date)
								AND ISCMS.PORTFOLIO_CODE = :investor_code
								AND ISCMS.FRACTION_TOTAL_SHARE > 0
								AND MWINSTDET.INSTRUMENT_DETAILS_ID =
									LSID.INSTRUMENT_DETAILS_ID) STOCK_INFO
							ON STOCK_INFO.PORTFOLIO_CODE = INDV.PORTFOLIO_CODE
							LEFT JOIN LS_TEMPLATE LTEMP
							ON LTEMP.TEMPLATE_ID = INDV.TEMPLATE_ID
					WHERE INDV.PORTFOLIO_CODE =  :investor_code
					ORDER BY INDV.PORTFOLIO_CODE, MARGIN_OR_NONMARGIN_FLAG DESC, SYMBOL ASC
            ";
        }

        $portfolioStatementResult = DB::select(
            $portfolioStatementSql,
            [
                'business_date' => $requestDate->format('Y-m-d'),
                'investor_code' => $user->investor_code,
                // 'investor_code' => 'B0286',
            ]
        );

		// $compareDate
        // dd($requestDate->format('Y-m-d'), $portfolioStatementResult);

		$accountStatus = $this->portfolioStatementAccountStatus($user->investor_code, $requestDate, $compareDate);
		
		// dd($accountStatus);


		$data['requestDate']               = $requestDate;
		$data['compareDate']               = $compareDate;
		$data['accountStatus']             = $accountStatus;
		$data['allCostAmount']             = collect($portfolioStatementResult)->where('total_share', '>', 0)->sum('total_cost_amount');
		$data['marginable_securities']     = collect($portfolioStatementResult)->where('margin_or_nonmargin_flag', 'y')->where('total_share', '>', 0)->values()->all();
		$data['non_marginable_securities'] = collect($portfolioStatementResult)->where('margin_or_nonmargin_flag', 'n')->where('total_share', '>', 0)->values()->all();


		$reportName = 'PortfolioStatement_' . $requestDate->format('d_m_Y') . '.pdf';
        $pdf = Pdf::loadView('portfolio.index', $data);
        $pdf->setPaper('a4');
        return $pdf->save($reportName)->stream($reportName);
    }

	private function portfolioStatementAccountStatus($portfolioCode, $requestDate, $compareDate)
	{
		if($portfolioCode == 'O01') {
			$accountStatusSql 	= "
			SELECT 
				NVL((P_PORTFOLIO_STATEMENT_TEMP.CURRENT_BALANCE), 0) 					AS CURRENT_BALANCE,
				NVL((P_PORTFOLIO_STATEMENT_TEMP.RECEIVABLES_SALES), 0) 					AS RECEIVABLES_SALES,
				NVL((P_PORTFOLIO_STATEMENT_TEMP.RECEIVABLE_DIVIDEND), 0) 				AS CASH_DIV_RECEIVED,
				NVL((P_PORTFOLIO_STATEMENT_TEMP.ACCRUED_FEES_CHARGE), 0) 				AS ACCRUED,
				NVL (
				   (  P_PORTFOLIO_STATEMENT_TEMP.ACCRUED_FEES_CHARGE
					+ P_PORTFOLIO_STATEMENT_TEMP.ACCRUED_MANAGEMENT_FEES),
				   0) 																	AS ACCRUED,
				NVL((P_PORTFOLIO_STATEMENT_TEMP.EQUITY_TO_DEBT_RATIO), 0) 				AS EQUITY_TO_DEBT_RATIO,
				NVL((P_PORTFOLIO_STATEMENT_TEMP.TOTAL_DEPOSIT), 0) 						AS TOTAL_DEPOSIT,
				NVL((P_PORTFOLIO_STATEMENT_TEMP.FUND_TRANSFER_IN), 0) 					AS TRANSFERIN,
				NVL((P_PORTFOLIO_STATEMENT_TEMP.REALIZE_GAIN_LOSS), 0) 					AS REALIZE_GAIN_LOSS,
				NVL((P_PORTFOLIO_STATEMENT_TEMP.DIVIDEND_INCOME), 0) 					AS DIVIDEND_INCOME,
				NVL((P_PORTFOLIO_STATEMENT_TEMP.WITHDRAW), 0) 							AS WITHDRAW,
				NVL((P_PORTFOLIO_STATEMENT_TEMP.FUND_TRANSFER_OUT), 0) 					AS TRANSFEROUT,
				NVL((P_PORTFOLIO_STATEMENT_TEMP.MARKET_VALUE_SECURITIES), 0) 			AS MKT_VALUE_SEC,
				NVL((P_PORTFOLIO_STATEMENT_TEMP.IPO_APPLICATION), 0) 					AS IPO_APPLICATION,
				NVL((P_PORTFOLIO_STATEMENT_TEMP.PRE_IPO_APPLICATION), 0) 				AS PRE_IPO_APPLICATION,
				NVL((P_PORTFOLIO_STATEMENT_TEMP.PREFERENCE_APPLICATION), 0) 			AS PREFERENCE_APPLICATION,
				NVL((P_PORTFOLIO_STATEMENT_TEMP.RIGHT_APPLICATION), 0) 					AS RIGHT_APPLICATION,
				NVL((P_PORTFOLIO_STATEMENT_TEMP.FUND_WITHDRAWAL_ORDER), 0) 				AS FUND_WITHDRAWAL_ORDER,
				NVL((P_PORTFOLIO_STATEMENT_TEMP.EQUITY), 0) 							AS EQUITY,
				NVL((P_PORTFOLIO_STATEMENT_TEMP.MARGIN_EQUITY), 0) 						AS MARGIN_EQUITY,
				NVL((P_PORTFOLIO_STATEMENT_TEMP.LOAN_RATIO), 0) 						AS LOAN_RATIO,
				(CASE
						WHEN COST_VALUE_SECURITIES > MARKET_VALUE_SECURITIES
						THEN
							((EQUITY * (1 + LOAN_RATIO)) - MARKET_VALUE_SECURITIES)
						ELSE
							((EQUITY * (1 + LOAN_RATIO)) - COST_VALUE_SECURITIES)
					END)
					AS PURCHASE_POWER,
				NVL((P_PORTFOLIO_STATEMENT_TEMP.PURCHASE_POWER), 0) 					AS PURCHASE_POWER1,
				NVL((P_PORTFOLIO_STATEMENT_TEMP.NETTING_STATUS), 0) 					AS NETTING_STATUS,
				NVL((P_PORTFOLIO_STATEMENT_TEMP.UNREALIZE_GAIN_LOSS), 0) 				AS UNREALIZE_GL,
				NVL((P_PORTFOLIO_STATEMENT_TEMP.RECEIVED_DIVIDEND), 0) 					AS RECEIVED_DIVIDEND,
				NVL((P_PORTFOLIO_STATEMENT_TEMP.RETURN_ON_INVESTMENT), 0) 				AS ROI,
				NVL((P_PORTFOLIO_STATEMENT_TEMP.MATURE_MARKET_VALUE_SECURITIES), 0) 	AS MATURE_MKT_VL_SEC,
				NVL((P_PORTFOLIO_STATEMENT_TEMP.CURRENT_ASSET_LIABILITIES), 0) 			AS CURRENT_ASSET_LIABILITIES,
				NVL((P_PORTFOLIO_STATEMENT_TEMP.LEDGER_BALANCE), 0) 					AS LEDGER_BALANCE,
				NVL((P_PORTFOLIO_STATEMENT_TEMP.AV_OF_PORTFOLIO), 0) 					AS AV_OF_PORTFOLIO,
				NVL((P_PORTFOLIO_STATEMENT_TEMP.MATURE_MARKET_VALUE_OF_SEC), 0) 		AS MATURE_MKT_VALUE_SEC,
				NVL((P_PORTFOLIO_STATEMENT_TEMP.COST_VALUE_SECURITIES), 0) 				AS COST_VALUE_SECURITIES,
				NVL((P_PORTFOLIO_STATEMENT_TEMP.RECEIVABLES_CURRENT_ASSET), 0) 			AS RECEIVABLES_CURRENT_ASSET,
				NVL((P_PORTFOLIO_STATEMENT_TEMP.INVEST_NL_SHARE), 0) 					AS INVEST_NL_SHARE,
				NVL((P_PORTFOLIO_STATEMENT_TEMP.RECEIVABLES_FDR), 0) 					AS RECEIVABLES_FDR,
				NVL((P_PORTFOLIO_STATEMENT_TEMP.RECEIVABLES_OTHER_ADVANCE), 0) 			AS RECEIVABLES_OTHER_ADVANCE,
				NVL((P_PORTFOLIO_STATEMENT_TEMP.RECEIVABLES_ADVANCE_TAX), 0) 			AS RECEIVABLES_ADVANCE_TAX,
				NVL((P_PORTFOLIO_STATEMENT_TEMP.RECEIVABLES_DIV_TAX_AMOUNT), 0) 		AS RECEIVABLES_DIV_TAX_AMOUNT,
				NVL((P_PORTFOLIO_STATEMENT_TEMP.PAYABLES), 0) 							AS PAYABLES,
				NVL((P_PORTFOLIO_STATEMENT_TEMP.DEDUCTED_MANAGEMENT_FEE), 0) 			AS DEDUCTED_MANAGEMENT_FEE,
				NVL((P_PORTFOLIO_STATEMENT_TEMP.DEDUCTED_MARGIN_LOAN), 0) 				AS DEDUCTED_MARGIN_LOAN,
				NVL((P_PORTFOLIO_STATEMENT_TEMP.RIGHT_SHARE_APPLIED), 0) 				AS RIGHT_SHARE_APPLIED,
				NVL((P_PORTFOLIO_STATEMENT_TEMP.IS_MARGIN_LOAN_ACTIVE), 0) 				AS IS_MARGIN_LOAN_ACTIVE,
				NVL((P_PORTFOLIO_STATEMENT_TEMP.RECEIVABLE_DIVIDEND), 0) 				AS CASH_DIV_RECEIVABLE,
				0 																		AS COUPON_RECEIVABLE
			FROM P_INVESTOR_PORTFOLIO_STATEMENT P_PORTFOLIO_STATEMENT_TEMP, IS_INVESTOR_DETAILS INVDET
			WHERE P_PORTFOLIO_STATEMENT_TEMP.PORTFOLIO_CODE = INVDET.PORTFOLIO_CODE
			AND INVDET.PORTFOLIO_CODE = :portfolio_code
			AND P_PORTFOLIO_STATEMENT_TEMP.BUSINESS_DATE =
			   (SELECT MAX(P_PORTFOLIO_STATEMENT_TEMP.BUSINESS_DATE)
				  FROM P_INVESTOR_PORTFOLIO_STATEMENT P_PORTFOLIO_STATEMENT_TEMP, L_BUSINESS_DATE BDAT
				 WHERE P_PORTFOLIO_STATEMENT_TEMP.PORTFOLIO_CODE = INVDET.PORTFOLIO_CODE
					   AND P_PORTFOLIO_STATEMENT_TEMP.BUSINESS_DATE = BDAT.BUSINESS_DATE
				   AND INVDET.PORTFOLIO_CODE = :portfolio_code
				   AND P_PORTFOLIO_STATEMENT_TEMP.BUSINESS_DATE <= :business_date)
			";
		} else {
			if($compareDate->gte($requestDate)) {
				$accountStatusSql 	= "
				SELECT 
					NVL((P_PORTFOLIO_STATEMENT_TEMP.CURRENT_BALANCE), 0) 				AS CURRENT_BALANCE,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.RECEIVABLES_SALES), 0) 				AS RECEIVABLES_SALES,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.RECEIVABLE_DIVIDEND), 0) 			AS CASH_DIV_RECEIVED,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.ACCRUED_FEES_CHARGE), 0) 			AS ACCRUED,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.EQUITY_TO_DEBT_RATIO), 0) 			AS EQUITY_TO_DEBT_RATIO,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.TOTAL_DEPOSIT), 0) 					AS TOTAL_DEPOSIT,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.FUND_TRANSFER_IN), 0) 				AS TRANSFERIN,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.REALIZE_GAIN_LOSS), 0) 				AS REALIZE_GAIN_LOSS,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.DIVIDEND_INCOME), 0) 				AS DIVIDEND_INCOME,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.WITHDRAW), 0) 						AS WITHDRAW,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.FUND_TRANSFER_OUT), 0) 				AS TRANSFEROUT,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.MARKET_VALUE_SECURITIES), 0) 		AS MKT_VALUE_SEC,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.IPO_APPLICATION), 0) 				AS IPO_APPLICATION,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.PRE_IPO_APPLICATION), 0) 			AS PRE_IPO_APPLICATION,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.PREFERENCE_APPLICATION), 0) 		AS PREFERENCE_APPLICATION,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.RIGHT_APPLICATION), 0) 				AS RIGHT_APPLICATION,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.FUND_WITHDRAWAL_ORDER), 0) 			AS FUND_WITHDRAWAL_ORDER,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.EQUITY), 0) 						AS EQUITY,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.MARGIN_EQUITY), 0) 					AS MARGIN_EQUITY,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.LOAN_RATIO), 0) 					AS LOAN_RATIO,
					(CASE
							WHEN COST_VALUE_SECURITIES > MARKET_VALUE_SECURITIES
							THEN
								((EQUITY * (1 + LOAN_RATIO)) - MARKET_VALUE_SECURITIES)
							ELSE
								((EQUITY * (1 + LOAN_RATIO)) - COST_VALUE_SECURITIES)
						END)
						AS PURCHASE_POWER,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.PURCHASE_POWER), 0) 					AS PURCHASE_POWER1,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.NETTING_STATUS), 0) 					AS NETTING_STATUS,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.UNREALIZE_GAIN_LOSS), 0) 				AS UNREALIZE_GL,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.RECEIVED_DIVIDEND), 0) 					AS RECEIVED_DIVIDEND,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.RETURN_ON_INVESTMENT), 0) 				AS ROI,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.MATURE_MARKET_VALUE_SECURITIES), 0) 	AS MATURE_MKT_VL_SEC,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.CURRENT_ASSET_LIABILITIES), 0) 			AS CURRENT_ASSET_LIABILITIES,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.LEDGER_BALANCE), 0) 					AS LEDGER_BALANCE,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.AV_OF_PORTFOLIO), 0) 					AS AV_OF_PORTFOLIO,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.MATURE_MARKET_VALUE_OF_SEC), 0) 		AS MATURE_MKT_VALUE_SEC,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.COST_VALUE_SECURITIES), 0) 				AS COST_VALUE_SECURITIES,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.RECEIVABLES_CURRENT_ASSET), 0) 			AS RECEIVABLES_CURRENT_ASSET,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.INVEST_NL_SHARE), 0) 					AS INVEST_NL_SHARE,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.RECEIVABLES_FDR), 0) 					AS RECEIVABLES_FDR,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.RECEIVABLES_OTHER_ADVANCE), 0) 			AS RECEIVABLES_OTHER_ADVANCE,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.RECEIVABLES_ADVANCE_TAX), 0) 			AS RECEIVABLES_ADVANCE_TAX,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.RECEIVABLES_DIV_TAX_AMOUNT), 0) 		AS RECEIVABLES_DIV_TAX_AMOUNT,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.PAYABLES), 0) 							AS PAYABLES,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.DEDUCTED_MANAGEMENT_FEE), 0) 			AS DEDUCTED_MANAGEMENT_FEE,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.DEDUCTED_MARGIN_LOAN), 0) 				AS DEDUCTED_MARGIN_LOAN,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.RIGHT_SHARE_APPLIED), 0) 				AS RIGHT_SHARE_APPLIED,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.IS_MARGIN_LOAN_ACTIVE), 0) 				AS IS_MARGIN_LOAN_ACTIVE,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.RECEIVABLE_DIVIDEND), 0) 				AS CASH_DIV_RECEIVABLE,
					0 																		AS COUPON_RECEIVABLE
				FROM P_PORTFOLIO_STATEMENT_TEMP, IS_INVESTOR_DETAILS INVDET
				WHERE P_PORTFOLIO_STATEMENT_TEMP.PORTFOLIO_CODE = INVDET.PORTFOLIO_CODE
				AND INVDET.PORTFOLIO_CODE = :portfolio_code
				AND P_PORTFOLIO_STATEMENT_TEMP.BUSINESS_DATE =
				   (SELECT MAX(P_PORTFOLIO_STATEMENT_TEMP.BUSINESS_DATE)
					  FROM P_PORTFOLIO_STATEMENT_TEMP P_PORTFOLIO_STATEMENT_TEMP, L_BUSINESS_DATE BDAT
					 WHERE P_PORTFOLIO_STATEMENT_TEMP.PORTFOLIO_CODE = INVDET.PORTFOLIO_CODE
					   AND P_PORTFOLIO_STATEMENT_TEMP.BUSINESS_DATE = BDAT.BUSINESS_DATE
					   AND INVDET.PORTFOLIO_CODE = :portfolio_code
					   AND P_PORTFOLIO_STATEMENT_TEMP.BUSINESS_DATE <= :business_date)
				--AND P_PORTFOLIO_STATEMENT_TEMP.BUSINESS_DATE = :business_date
				";
			} else {
				$accountStatusSql 	= "
				SELECT 
					NVL((P_PORTFOLIO_STATEMENT_TEMP.CURRENT_BALANCE), 0) 				AS CURRENT_BALANCE,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.RECEIVABLES_SALES), 0) 				AS RECEIVABLES_SALES,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.RECEIVABLE_DIVIDEND), 0) 			AS CASH_DIV_RECEIVED,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.ACCRUED_FEES_CHARGE), 0) 			AS ACCRUED,
					NVL (
				   (  P_PORTFOLIO_STATEMENT_TEMP.ACCRUED_FEES_CHARGE
					+ P_PORTFOLIO_STATEMENT_TEMP.ACCRUED_MANAGEMENT_FEES),
				   0) 																	AS ACCRUED,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.EQUITY_TO_DEBT_RATIO), 0) 			AS EQUITY_TO_DEBT_RATIO,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.TOTAL_DEPOSIT), 0) 					AS TOTAL_DEPOSIT,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.FUND_TRANSFER_IN), 0) 				AS TRANSFERIN,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.REALIZE_GAIN_LOSS), 0) 				AS REALIZE_GAIN_LOSS,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.DIVIDEND_INCOME), 0) 				AS DIVIDEND_INCOME,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.WITHDRAW), 0) 						AS WITHDRAW,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.FUND_TRANSFER_OUT), 0) 				AS TRANSFEROUT,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.MARKET_VALUE_SECURITIES), 0) 		AS MKT_VALUE_SEC,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.IPO_APPLICATION), 0) 				AS IPO_APPLICATION,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.PRE_IPO_APPLICATION), 0) 			AS PRE_IPO_APPLICATION,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.PREFERENCE_APPLICATION), 0) 		AS PREFERENCE_APPLICATION,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.RIGHT_APPLICATION), 0) 				AS RIGHT_APPLICATION,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.FUND_WITHDRAWAL_ORDER), 0) 			AS FUND_WITHDRAWAL_ORDER,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.EQUITY), 0) 						AS EQUITY,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.MARGIN_EQUITY), 0) 					AS MARGIN_EQUITY,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.LOAN_RATIO), 0) 					AS LOAN_RATIO,
					(CASE
							WHEN COST_VALUE_SECURITIES > MARKET_VALUE_SECURITIES
							THEN
								((EQUITY * (1 + LOAN_RATIO)) - MARKET_VALUE_SECURITIES)
							ELSE
								((EQUITY * (1 + LOAN_RATIO)) - COST_VALUE_SECURITIES)
						END)
						AS PURCHASE_POWER,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.PURCHASE_POWER), 0) 					AS PURCHASE_POWER1,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.NETTING_STATUS), 0) 					AS NETTING_STATUS,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.UNREALIZE_GAIN_LOSS), 0) 				AS UNREALIZE_GL,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.RECEIVED_DIVIDEND), 0) 					AS RECEIVED_DIVIDEND,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.RETURN_ON_INVESTMENT), 0) 				AS ROI,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.MATURE_MARKET_VALUE_SECURITIES), 0) 	AS MATURE_MKT_VL_SEC,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.CURRENT_ASSET_LIABILITIES), 0) 			AS CURRENT_ASSET_LIABILITIES,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.LEDGER_BALANCE), 0) 					AS LEDGER_BALANCE,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.AV_OF_PORTFOLIO), 0) 					AS AV_OF_PORTFOLIO,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.MATURE_MARKET_VALUE_OF_SEC), 0) 		AS MATURE_MKT_VALUE_SEC,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.COST_VALUE_SECURITIES), 0) 				AS COST_VALUE_SECURITIES,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.RECEIVABLES_CURRENT_ASSET), 0) 			AS RECEIVABLES_CURRENT_ASSET,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.INVEST_NL_SHARE), 0) 					AS INVEST_NL_SHARE,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.RECEIVABLES_FDR), 0) 					AS RECEIVABLES_FDR,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.RECEIVABLES_OTHER_ADVANCE), 0) 			AS RECEIVABLES_OTHER_ADVANCE,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.RECEIVABLES_ADVANCE_TAX), 0) 			AS RECEIVABLES_ADVANCE_TAX,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.RECEIVABLES_DIV_TAX_AMOUNT), 0) 		AS RECEIVABLES_DIV_TAX_AMOUNT,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.PAYABLES), 0) 							AS PAYABLES,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.DEDUCTED_MANAGEMENT_FEE), 0) 			AS DEDUCTED_MANAGEMENT_FEE,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.DEDUCTED_MARGIN_LOAN), 0) 				AS DEDUCTED_MARGIN_LOAN,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.RIGHT_SHARE_APPLIED), 0) 				AS RIGHT_SHARE_APPLIED,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.IS_MARGIN_LOAN_ACTIVE), 0) 				AS IS_MARGIN_LOAN_ACTIVE,
					NVL((P_PORTFOLIO_STATEMENT_TEMP.RECEIVABLE_DIVIDEND), 0) 				AS CASH_DIV_RECEIVABLE,
					COUPON_DETAILS.COUPON_RECEIVABLE 										AS COUPON_RECEIVABLE
				FROM P_INVESTOR_PORTFOLIO_STATEMENT P_PORTFOLIO_STATEMENT_TEMP, IS_INVESTOR_DETAILS INVDET
				LEFT JOIN (SELECT INVDET.PORTFOLIO_CODE     AS PORTFOLIO_CODE,
							   TB.CB_AMOUNT              AS COUPON_RECEIVABLE
						  FROM IS_PORTFOLIO_TRIAL_BAL TB, IS_INVESTOR_DETAILS INVDET
						 WHERE     TB.AC_CODE = '305009003'
							   AND TB.PORTFOLIO_CODE = INVDET.PORTFOLIO_CODE
							   AND TB.BALANCE_DATE =
								   (SELECT MAX (MTB.BALANCE_DATE)
									  FROM IS_PORTFOLIO_TRIAL_BAL MTB
									 WHERE     MTB.AC_CODE = TB.AC_CODE
										   AND MTB.PORTFOLIO_CODE = TB.PORTFOLIO_CODE
										   AND MTB.BALANCE_DATE <= :business_date)) COUPON_DETAILS
											   ON COUPON_DETAILS.PORTFOLIO_CODE = INVDET.PORTFOLIO_CODE
				WHERE P_PORTFOLIO_STATEMENT_TEMP.PORTFOLIO_CODE = INVDET.PORTFOLIO_CODE
				AND INVDET.PORTFOLIO_CODE = :portfolio_code
				AND P_PORTFOLIO_STATEMENT_TEMP.BUSINESS_DATE =
				   (SELECT MAX(P_PORTFOLIO_STATEMENT_TEMP.BUSINESS_DATE)
					  FROM P_INVESTOR_PORTFOLIO_STATEMENT P_PORTFOLIO_STATEMENT_TEMP, L_BUSINESS_DATE BDAT
					 WHERE P_PORTFOLIO_STATEMENT_TEMP.PORTFOLIO_CODE = INVDET.PORTFOLIO_CODE
					   AND P_PORTFOLIO_STATEMENT_TEMP.BUSINESS_DATE = BDAT.BUSINESS_DATE
					   AND INVDET.PORTFOLIO_CODE = :portfolio_code
					   AND P_PORTFOLIO_STATEMENT_TEMP.BUSINESS_DATE <= :business_date)
				";
			}
		}

        $accountStatusResult = DB::select(
            $accountStatusSql,
            [
                'business_date' => $requestDate->format('Y-m-d'),
                'portfolio_code' => $portfolioCode,
                // 'investor_code' => 'B0286',
            ]
        );

		return collect($accountStatusResult)->first();
	}
}
