<?php

namespace App\Http\Controllers\Withdraw;

use Carbon\Carbon;
use App\Models\BusinessDate;
use App\Models\MessageRoute;
use Illuminate\Http\Request;
use App\Models\TransactionDetails;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class WithdrawController extends Controller
{
    public function availableBalance()
    {
        $user         = Auth::user();
        $businessDate = BusinessDate::latest('business_date')->first();

        $availableBalanceSql = "
            SELECT 
                NVL(
                NVL (A.CURRENT_BALANCE, 0) - NVL (B.WITHDRAW_AMOUNT, 0), 
                0
                ) BALANCE 
            FROM 
                P_PORTFOLIO_STATEMENT_TEMP A, 
                (
                SELECT 
                    PORTFOLIO_CODE, 
                    SUM (AMOUNT) WITHDRAW_AMOUNT 
                FROM 
                    IS_TRANSACTION_DETAILS 
                WHERE 
                    HONOR_DISHONOR_CANCEL = 'p' 
                    AND TRANSACTION_TYPE = 'withdraw' 
                GROUP BY 
                    PORTFOLIO_CODE
                ) B 
            WHERE 
                A.PORTFOLIO_CODE = B.PORTFOLIO_CODE(+) 
                AND A.PORTFOLIO_STATEMENT_ID = (
                SELECT 
                    MAX (
                    PORTFOLIO_STATEMENT_ID
                    ) 
                FROM 
                    P_PORTFOLIO_STATEMENT_TEMP 
                WHERE 
                    PORTFOLIO_CODE = :investor_code
                )
      
        ";

        $availableBalanceResult = DB::select(
            $availableBalanceSql,
            [
                'investor_code' => $user->investor_code,
            ]
        );

        $availableBalance = collect($availableBalanceResult)->first()->balance;

        return response()->json([
            'status'            => 'success',
            'available_balance' => number_format($availableBalance, 2),
        ]);
    }

    public function withdrawList()
    {
        $user         = Auth::user();
        $businessDate = BusinessDate::latest('business_date')->first();
        
        $withdrawListSql = "
            SELECT 
                B.ORG_NAME, 
                CASE A.PAYMENT_TYPE WHEN 'cheque' THEN 'BEFTN' WHEN 'po' THEN 'Pay Order' WHEN 'dd' THEN 'Demand Draft' WHEN 'cd' THEN 'Cash Deposit' WHEN 'ft' THEN 'Fund Transfer' ELSE 'N/A' END AS PAYMENT_TYPE, 
                CASE A.MSG_STATUS WHEN 'approved' THEN 'Approved' WHEN 'maked' THEN 'Pending' WHEN 'checked' THEN 'Pending' ELSE 'Canceled' END AS STATUS, 
                A.CHQ_NO, 
                A.CHQ_DATE, 
                A.AMOUNT, 
                A.BUSINESS_DATE, 
                A.RECORD_DATE 
            FROM 
                IS_TRANSACTION_DETAILS A, 
                GS_MONEY_MKT_ORG B 
            WHERE 
                    A.ORG_ID           = B.ORG_ID(+)
                AND A.TRANSACTION_TYPE = 'withdraw'
                and a.PORTFOLIO_CODE   = :investor_code
                AND A.BUSINESS_DATE    = :business_date
            ORDER BY 
                A.TRANSACTION_DETAILS_ID DESC FETCH FIRST 10 ROWS ONLY      
        ";

        $withdrawListResult = DB::select(
            $withdrawListSql,
            [
                'business_date' => $businessDate->business_date,
                'investor_code' => $user->investor_code,
            ]
        );

        $withdraw_list = [];
        foreach ($withdrawListResult as $index => $value) {
            $withdraw_list[$index]['org_name']      = $value->org_name;
            $withdraw_list[$index]['payment_type']  = $value->payment_type;
            $withdraw_list[$index]['chq_no']        = $value->chq_no;
            $withdraw_list[$index]['amount']        = number_format($value->amount, 2);
            $withdraw_list[$index]['status']        = $value->status;
            $withdraw_list[$index]['chq_date']      = Carbon::createFromFormat('Y-m-d H:i:s', $value->chq_date)->format('d-M-Y');
            $withdraw_list[$index]['business_date'] = Carbon::createFromFormat('Y-m-d H:i:s', $value->business_date)->format('d-M-Y');
            $withdraw_list[$index]['record_date']   = Carbon::createFromFormat('Y-m-d H:i:s', $value->record_date)->format('d-M-Y H:i:s a');

        }

        return response()->json(
            [
                'status'        => 'success',
                'withdraw_list' => $withdraw_list,
            ]
        );
    }

    public function submitWithdraw(Request $request)
    {


        $availableBalance = (float)str_replace(',', '', $this->availableBalance()->getData()->available_balance);
        $maxWidrawAmount = ($availableBalance - 1000);


        $validator = Validator::make(
            $request->all(),
            [
                'withdraw_amount'        => "required|numeric|max:$maxWidrawAmount",
                // 'deposit_account_id'    => 'required|exists:gs_account_details,account_details_id',
                // 'payment_type'          => 'required|in:cheque,po,dd,cd,ft',
                // 'deposit_instrument_no' => 'required',
                // 'bank_id'               => 'required|exists:gs_money_mkt_org,org_id',
                // 'branch_id'             => 'required|exists:gs_org_branch,org_branch_id',
                // 'deposit_date'          => 'required|date_format:d-m-Y',
                // 'deposit_slip'          => 'required|file|extensions:jpg,png,jpeg,pdf',
            ],
            [
                'withdraw_amount.required'          => 'The withdraw amount filed is required',
                'withdraw_amount.numeric'           => 'The withdraw amount must be a number',
                'withdraw_amount.max'               => "The withdraw amount must be equal or less than $maxWidrawAmount",
                // 'deposit_account_id.required'    => 'The deposit account filed name is required',
                // 'deposit_account_id.exists'      => 'The deposit account is invalid',
                // 'payment_type.required'          => 'The payment type filed is required',
                // 'payment_type.in'                => 'The payment type is invalid',
                // 'deposit_instrument_no.required' => 'The deposit instrument number filed is required',
                // 'bank_id.required'               => 'The bank name filed is required',
                // 'bank_id.exists'                 => 'The bank name is invalid',
                // 'branch_id.required'             => 'The branch name filed is required',
                // 'branch_id.exists'               => 'The branch name is invalid',
                // 'deposit_date.required'          => 'The deposit instrument date filed is required',
                // 'deposit_date.date_format'       => 'The deposit instrument date field must be in d-m-Y format',
                // 'deposit_slip.required'          => 'The deposit slip filed is required',
                // 'deposit_slip.file'              => 'The deposit slip must be a file',
                // 'deposit_slip.extensions'        => 'The deposit slip must be in jpg, png, jpeg or pdf format',
            ]
        );

        if ($validator->fails()) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Form validation error',
                'errors' => $validator->errors()->messages(),
            ], 400);
        } else {
            try {
                $user                 = Auth::user();
                $businessDate         = BusinessDate::latest('business_date')->first();

                $transactionDetails                        = new TransactionDetails();
                $transactionDetails->org_id                = 4;
                // $transactionDetails->branch_id             = $request->branch_id;
                $transactionDetails->portfolio_code        = $user->investor_code;
                $transactionDetails->transaction_type      = 'withdraw';
                $transactionDetails->payment_type          = 'cheque';
                $transactionDetails->chq_no                = 100000000000001;
                $transactionDetails->chq_date              = now();
                $transactionDetails->amount                = $request->withdraw_amount;
                $transactionDetails->remarks               = 'web withdraw';
                $transactionDetails->operate_by            = 0;
                $transactionDetails->msg_status            = 'maked';
                $transactionDetails->account_details_id    = 482;
                $transactionDetails->file_upload           = '';
                $transactionDetails->send_to_bank          = 'n';
                $transactionDetails->honor_dishonor_cancel = 'p';
                $transactionDetails->cheque_print_flag     = 'n';
                $transactionDetails->business_date         = $businessDate->business_date;
                $transactionDetails->record_date           = now();

                if ($transactionDetails->save()) {
                    $messageRoute                         = new MessageRoute();
                    $messageRoute->send_by                = '4772';
                    $messageRoute->receive_by             = '4772';
                    $messageRoute->viewableto             = 4;
                    $messageRoute->transaction_details_id = $transactionDetails->transaction_details_id;
                    $messageRoute->remarks                = 'Withdraw by Investor';
                    $messageRoute->status                 = 'N/A';
                    $messageRoute->business_date          = $businessDate->business_date;
                    $messageRoute->record_date            = now();
                    $messageRoute->send_date              = now();

                    $messageRoute->save();

                    return response()->json([
                        'status'      => 'success',
                        'message'   => 'Withdraw submitted successfully',
                    ]);
                }
            } catch (\Throwable $th) {
                return response()->json([
                    'status'      => 'error',
                    'dev_message' => $th->getMessage(),
                    'message'      => 'Withdraw form submit unsuccessful',
                ], 400);
            }
        }
    }
}
