<?php

namespace App\Http\Controllers\Report;

use Carbon\Carbon;
use Illuminate\Http\Request;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use App\Models\FiscalYear;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class TaxCertificateController extends Controller
{
    public function fiscalYear() {
        $fiscalYear = FiscalYear::all(['fiscal_year_id', 'fiscal_year']);
        
        return response()->json([
            'status'      => 'success',
            'fiscal_year' => $fiscalYear,
        ]);
    }

    public function taxCertificateDownload(Request $request)
    {
        $validator = Validator::make(
            $request->all(),
            [
                'fiscal_year_id' => 'required',
            ],
            [
                'fiscal_year_id.required' => 'The fiscal year field is required',
            ]
        );

        if ($validator->fails()) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Form validation error',
                'errors' => $validator->errors()->messages(),
            ], 400);
        }

        $data = [
            'title' => 'Tax Certificate',
        ];

        $fiscalYear = FiscalYear::find($request->fiscal_year_id);

        $user                    = Auth::user();
		$investorDetail          = $user->investorDetail;		
		$toDate                  = Carbon::createFromFormat('Y-m-d H:i:s', $fiscalYear->fiscal_start);
        $fromDate                = Carbon::createFromFormat('Y-m-d H:i:s', $fiscalYear->fiscal_end);
        $compareDate             = Carbon::createFromFormat('Y-m-d', '2022-12-31');
		$data['account_type']    = $investorDetail->templateDetails->product_short_name;
		$data['account_no']      = $investorDetail->portfolio_code;
		$data['bo_no']           = $investorDetail->bo_account_no;
		$data['account_date']    = Carbon::createFromFormat('Y-m-d H:i:s', $investorDetail->business_date)->format(env('OUTPUT_DATE_FORMAT'));
		$data['account_name']    = $investorDetail->investor_name;
		$data['account_address'] = $investorDetail->mailing_address;
        $data['report_date']     = $fromDate->format('d-m-Y') . ' To ' . $toDate->format('d-m-Y');

        if ($investorDetail->account_enable_disable == 'e') {
			$data['account_status'] = 'Active';
		}

		if ($investorDetail->account_close == 'y') {
			$data['account_status'] = 'Close';
		}

		if ($investorDetail->account_suspend == 'y') {
			$data['account_status'] = 'Suspend';
		}

        // taxcertificate

        // dd($investorDetail);

        $data['operation_mode'] = $investorDetail->operation_mode_id;
        $data['father_name'] = $investorDetail->father_name;
        $data['mother_name'] = $investorDetail->mother_name;
        $data['Date'] = '';

        $divIncome      = $this->getCashDividendForInvCert($user->investor_code, $fromDate, $toDate);
        $totalDeposit   = $this->DepositeDuringPeriod($user->investor_code, $fromDate, $toDate);
        $totalWithDraw  = $this->getWithdrawDuringPeriod($user->investor_code, $fromDate, $toDate);
        $interestIncome = $this->getMarginInterest($user->investor_code, $fromDate, $toDate);
        $mgtFeePeriod   = $this->getManagementFeeCertificate($user->investor_code, $fromDate, $toDate);
        $cDBLCharge     = $this->getBOMaintenanceCharge($user->investor_code, $fromDate, $toDate);
        $rightAppAmt    = $this->getRightShareAppAmount($user->investor_code, $fromDate, $toDate);
        $ipoAppAmt      = $this->getIPOAppAmount($user->investor_code, $fromDate, $toDate);
        $realizedValue  = $this->getRealizGNValue($user->investor_code, $fromDate, $toDate);
        $yearlyCharge   = $this->getYearlyChargeForWithdraw($user->investor_code,$toDate);
        $investorData   = $this->getEquityData($user->investor_code, $fromDate);
        $investorClData  = $this->getEquityData($user->investor_code,$toDate);
        $getAvilBalance  		= $this->getClosingPortData($user->investor_code, $toDate, $compareDate);
        // dd($investorData);


        $data['totalCashDivAmt']     = is_null($divIncome) ? 0 : $divIncome->totalCashDivAmt;
        $data['totaltaxDivAmt']      = is_null($divIncome) ? 0 : $divIncome->totaltaxDivAmt;
        $data['totalnetDivAmt']      = is_null($divIncome) ? 0 : $divIncome->totalnetDivAmt;
        $data['openEquity']          = is_null($investorData) ? 0 : $investorData->equity;
        $data['closingEquity']       = is_null($getAvilBalance) ? 0 : $getAvilBalance->equity;
        $data['getBalanceDrCr']      = is_null($getAvilBalance) ? 0 : $getAvilBalance->current_balance;
        $data['getAccurFeesCrg']     = $interestIncome+$mgtFeePeriod+$yearlyCharge+$cDBLCharge;
        $data['openCostValOfSec']    = is_null($investorData) ? 0 : $investorData->cost_value_securities;
        $data['costValueSecurities'] = is_null($investorClData) ? 0 : $investorClData->cost_value_securities;
        $data['totalMarketPrice']    = is_null($investorClData) ? 0 : $investorClData->mkt_value_securities;
        $data['totalDeposit']        = $totalDeposit;
        $data['totalWithDraw']       = $totalWithDraw;
        $data['rightAppAmt']         = $rightAppAmt;
        $data['ipoAppAmt']           = $ipoAppAmt;
        $data['gainLossWithCOmm']    = is_null($realizedValue) ? 0 : $realizedValue->allGainLoss;
        // $data['openEquity']          = 154555;
        // $data['openCostValOfSec']    = 54215;
        // $data['totalMarketPrice']    = 585154;
        // $data['costValueSecurities'] = 85845;



        $reportName = 'TaxCertificate_' . $fromDate->format('d_m_Y') . '_' . $toDate->format('d_m_Y') . '.pdf';
        $pdf = Pdf::loadView('taxcertificate.index', $data);
        $pdf->setPaper('a4');
        return $pdf->save($reportName)->stream($reportName);
    }

    private function getCashDividendForInvCert($portfolioCode, $fromDate, $toDate)
    {
        $dividendSql = "
            SELECT 
                A.PORTFOLIO_CODE, 
                A.DIV_RCV_AMOUNT, 
                B.DIV_TAX_AMOUNT, 
                (
                    A.DIV_RCV_AMOUNT - B.DIV_TAX_AMOUNT
                ) AS NET_DIV_AMT 
            FROM 
                (
                    SELECT 
                        IPC.PORTFOLIO_CODE, 
                        SUM (IPC.AMOUNT) AS DIV_RCV_AMOUNT --, IPC.NTR
                    FROM 
                        IS_PORTFOLIO_MASTER IPM, 
                        IS_PORTFOLIO_CHILD IPC 
                    WHERE 
                        IPM.PORTFOLIO_MASTER_ID = IPC.PORTFOLIO_MASTER_ID --AND   IPC.NARRATION NOT LIKE 'Fraction dividend %'
                        AND IPC.PORTFOLIO_CODE = :portfolio_code
                        AND IPC.AC_CODE IN ('501001001') --       AND IPC.NTR = 'C'
                        AND IPM.CB_CODE IS NOT NULL 
                        AND IPM.TRAN_DATE BETWEEN :from_date AND :to_date
                        GROUP BY 
                        IPC.PORTFOLIO_CODE
                ) A 
                LEFT JOIN (
                    SELECT 
                        IPC.PORTFOLIO_CODE, 
                        SUM (IPC.AMOUNT) AS DIV_TAX_AMOUNT 
                    FROM 
                        IS_PORTFOLIO_MASTER IPM, 
                        IS_PORTFOLIO_CHILD IPC 
                    WHERE 
                        IPM.PORTFOLIO_MASTER_ID = IPC.PORTFOLIO_MASTER_ID --        AND   IPC.NARRATION LIKE 'Dividend %'
                        AND IPC.PORTFOLIO_CODE = :portfolio_code
                        AND IPC.AC_CODE IN ('601001013', '305011003') 
                        AND IPM.TRAN_DATE BETWEEN :from_date AND :to_date
                        GROUP BY 
                        IPC.PORTFOLIO_CODE
                ) B ON A.PORTFOLIO_CODE = B.PORTFOLIO_CODE
        ";

        $dividendSqlResult = DB::select(
            $dividendSql,
            [
                'from_date'      => $fromDate->format('Y-m-d'),
                'to_date'        => $toDate->format('Y-m-d'),
                'portfolio_code' => $portfolioCode,
            ]
        );

        $dividendSqlResultCollection = collect($dividendSqlResult);
        $data['totalCashDivAmt']     = $dividendSqlResultCollection->sum('div_rcv_amount');
        $data['totaltaxDivAmt']      = $dividendSqlResultCollection->sum('div_tax_amount');
        $data['totalnetDivAmt']      = $dividendSqlResultCollection->sum('net_div_amt');

        return ((object) $data);
    }

    private function DepositeDuringPeriod($portfolioCode, $fromDate, $toDate)
    {
        $depositSql = "
            SELECT 
                SUM(SIP.AMOUNT) as DEPOSITE 
            FROM 
                IS_TRANSACTION_DETAILS SIP, 
                IS_INVESTOR_DETAILS INDV 
            WHERE 
                INDV.PORTFOLIO_CODE = :portfolio_code
                AND SIP.BUSINESS_DATE BETWEEN :from_date AND :to_date 
                AND INDV.PORTFOLIO_CODE = SIP.PORTFOLIO_CODE 
                AND SIP.TRANSACTION_TYPE = 'deposit' 
                AND SIP.Send_To_Bank = 'y' 
                AND SIP.Honor_Dishonor_Cancel = 'h'

        ";

        $depositSqlResult = DB::select(
            $depositSql,
            [
                'from_date'      => $fromDate->format('Y-m-d'),
                'to_date'        => $toDate->format('Y-m-d'),
                'portfolio_code' => $portfolioCode,
            ]
        );

        $depositSqlResultCollection = collect($depositSqlResult)->first();

        return $depositSqlResultCollection->deposite;
    }

    private function getWithdrawDuringPeriod($portfolioCode, $fromDate, $toDate)
    {
        $withdrawSql = "
            SELECT 
                SUM(SIP.AMOUNT) as WITHDRAW 
            FROM 
                IS_TRANSACTION_DETAILS SIP, 
                IS_INVESTOR_DETAILS INDV 
            WHERE 
                INDV.PORTFOLIO_CODE = :portfolio_code
                AND SIP.BUSINESS_DATE BETWEEN :from_date AND :to_date 
                AND INDV.PORTFOLIO_CODE = SIP.PORTFOLIO_CODE 
                AND SIP.TRANSACTION_TYPE = 'withdraw' 
                AND SIP.Send_To_Bank = 'y' 
                AND SIP.Honor_Dishonor_Cancel = 'h'

        ";

        $withdrawSqlResult = DB::select(
            $withdrawSql,
            [
                'from_date'      => $fromDate->format('Y-m-d'),
                'to_date'        => $toDate->format('Y-m-d'),
                'portfolio_code' => $portfolioCode,
            ]
        );

        $withdrawSqlResultCollection = collect($withdrawSqlResult)->first();

        return $withdrawSqlResultCollection->withdraw;
    }

    private function getMarginInterest($portfolioCode, $fromDate, $toDate)
    {
        $interestSql = "
            SELECT 
                SUM(MLI.INTEREST_TODAY) AS TOT_MLI 
            FROM 
                P_MARGIN_LOAN_INTEREST MLI 
            WHERE 
                MLI.PORTFOLIO_CODE = :portfolio_code
                AND MLI.BUSINESS_DATE BETWEEN :from_date AND :to_date


        ";

        $interestSqlResult = DB::select(
            $interestSql,
            [
                'from_date'      => $fromDate->format('Y-m-d'),
                'to_date'        => $toDate->format('Y-m-d'),
                'portfolio_code' => $portfolioCode,
            ]
        );

        $interestSqlResultCollection = collect($interestSqlResult)->first();

        return $interestSqlResultCollection->tot_mli;
    }

    private function getManagementFeeCertificate($portfolioCode, $fromDate, $toDate)
    {
        $managementFeeSql = "
            SELECT 
                NVL(
                    SUM(
                    IS_MANAGEMENT_FEE.MANGEMENT_FEE_TODAY
                    ), 
                    0
                ) AS MFEE 
            FROM 
                IS_MANAGEMENT_FEE 
            WHERE 
                IS_MANAGEMENT_FEE.BUSINESS_DATE BETWEEN :from_date AND :to_date
                AND IS_MANAGEMENT_FEE.PORTFOLIO_CODE = :portfolio_code
                AND LOWER(
                    IS_MANAGEMENT_FEE.TRANSACTION_CLOSE
                ) = 'y'
        ";

        $managementFeeSqlResult = DB::select(
            $managementFeeSql,
            [
                'from_date'      => $fromDate->format('Y-m-d'),
                'to_date'        => $toDate->format('Y-m-d'),
                'portfolio_code' => $portfolioCode,
            ]
        );

        $managementFeeSqlResultCollection = collect($managementFeeSqlResult)->first();

        return $managementFeeSqlResultCollection->mfee;
    }

    private function getBOMaintenanceCharge($portfolioCode, $fromDate, $toDate)
    {
        $cdblFeeSql = "
            SELECT 
                SUM (
                    NVL (IPC.AMOUNT, 0)
                ) AS CHARGE_AMOUNT 
            FROM 
                IS_PORTFOLIO_MASTER IPM, 
                IS_PORTFOLIO_CHILD IPC 
            WHERE 
                IPC.AC_CODE = '601001024' 
                AND IPC.PORTFOLIO_CODE = :portfolio_code
                AND IPM.PORTFOLIO_MASTER_ID = IPC.PORTFOLIO_MASTER_ID
                AND IPM.TRAN_DATE BETWEEN :from_date AND :to_date
        ";

        $cdblFeeSqlResult = DB::select(
            $cdblFeeSql,
            [
                'from_date'      => $fromDate->format('Y-m-d'),
                'to_date'        => $toDate->format('Y-m-d'),
                'portfolio_code' => $portfolioCode,
            ]
        );

        $cdblFeeSqlResultCollection = collect($cdblFeeSqlResult)->first();

        return $cdblFeeSqlResultCollection->charge_amount;
    }

    private function getRightShareAppAmount($portfolioCode, $fromDate, $toDate)
    {
        $rightShareSql = "
            SELECT 
                NVL(
                    SUM(RSAPP.AMOUNT), 
                    0
                ) AS TOTALRSAPPLY 
            FROM 
                IS_RIGHT_SHARE_APPLICATION RSAPP 
            WHERE 
                RSAPP.PORTFOLIO_CODE = :portfolio_code
                AND RSAPP.STATUS = 'p' 
                AND RSAPP.BUSINESS_DATE BETWEEN :from_date AND :to_date
        ";

        $rightShareSqlResult = DB::select(
            $rightShareSql,
            [
                'from_date'      => $fromDate->format('Y-m-d'),
                'to_date'        => $toDate->format('Y-m-d'),
                'portfolio_code' => $portfolioCode,
            ]
        );

        $rightShareSqlResultCollection = collect($rightShareSqlResult)->first();

        return $rightShareSqlResultCollection->totalrsapply;
    }

    private function getIPOAppAmount($portfolioCode, $fromDate, $toDate)
    {
        $ipoSql = "
            SELECT 
                NVL(
                    SUM(PORTTB.TOTAL_AMOUNT), 
                    0
                ) AS TOTALIPOAPPLY 
            FROM 
                IS_IPO_APPLICATION PORTTB 
            WHERE 
                PORTTB.STATUS = 'p' 
                AND PORTTB.PORTFOLIO_CODE = :portfolio_code
                AND PORTTB.BUSINESS_DATE BETWEEN :from_date AND :to_date

        ";

        $ipoSqlResult = DB::select(
            $ipoSql,
            [
                'from_date'      => $fromDate->format('Y-m-d'),
                'to_date'        => $toDate->format('Y-m-d'),
                'portfolio_code' => $portfolioCode,
            ]
        );

        $ipoSqlResultCollection = collect($ipoSqlResult)->first();

        return $ipoSqlResultCollection->totalipoapply;
    }

    private function getRealizGNValue($portfolioCode, $fromDate, $toDate)
    {
        $realizeGainSql = "
            SELECT INDV.PORTFOLIO_CODE,
					 IS_CAP_MKT_STOCK.SHARES,
					 (CASE WHEN IS_CAP_MKT_STOCK.BUSINESS_DATE <= '31-Dec-2022' THEN 
					 (IS_CAP_MKT_STOCK.BROKER_COMMISSION+IS_CAP_MKT_STOCK.SATTLEMENT_FEE)
					 ELSE
						 IS_CAP_MKT_STOCK.BROKER_COMMISSION
					 END) AS BROKER_COMMISSION,
					 IS_CAP_MKT_STOCK.AVG_RATE,
					 IS_CAP_MKT_STOCK.ACTUAL_RATE,
					 IS_CAP_MKT_STOCK.TOTAL_AMOUNT,
					 IS_CAP_MKT_STOCK.BUSINESS_DATE,
					 IBRKCHG.CHARGE
				FROM IS_CAP_MKT_STOCK,
					 LS_INSTRUMENT_DETAILS LSID,
					 IS_INVESTOR_DETAILS  INDV,
					 IS_I_BROKER_CHARGE   IBRKCHG,
					 (SELECT MWINST.INSTRUMENT_DETAILS_ID, MWINST.SYMBOL
						FROM LS_MARKET_WISE_INSTRUMENT MWINST
					   WHERE MWINST.MARKET_DETAILS_ID = 61) MWINSTRUMENT
			   WHERE     LSID.INSTRUMENT_DETAILS_ID =
						 IS_CAP_MKT_STOCK.INSTRUMENT_DETAILS_ID
					 AND IS_CAP_MKT_STOCK.PORTFOLIO_CODE = :portfolio_code
					 AND IBRKCHG.PORTFOLIO_CODE = :portfolio_code
					 AND INDV.PORTFOLIO_CODE = :portfolio_code
					 AND IS_CAP_MKT_STOCK.I_BROKER_CHARGE_ID = IBRKCHG.I_BROKER_CHARGE_ID
					 AND IS_CAP_MKT_STOCK.STATUS = 'enc'
					 AND IS_CAP_MKT_STOCK.STOCK_MATURE_DATE IS NULL
					 AND IS_CAP_MKT_STOCK.BUSINESS_DATE BETWEEN :from_date AND :to_date
					 AND MWINSTRUMENT.INSTRUMENT_DETAILS_ID = LSID.INSTRUMENT_DETAILS_ID
			ORDER BY IS_CAP_MKT_STOCK.BUSINESS_DATE ASC
        ";

        $realizeGainSqlResult = DB::select(
            $realizeGainSql,
            [
                'from_date'      => $fromDate->format('Y-m-d'),
                'to_date'        => $toDate->format('Y-m-d'),
                'portfolio_code' => $portfolioCode,
            ]
        );

        // dd($realizeGainSqlResult);

        $allTotalBuy 			= 0;
        $allTotalSell 			= 0;
        $allGainLoss 			= 0;
        $indGLTotal 			= 0;
        foreach ($realizeGainSqlResult as $value) {            
            $indvShare     = $value->shares;
            $bCom          = $value->broker_commission;
            $indvAvgPrice  = $value->avg_rate;
            $totSellPrice  = $value->total_amount;
            $allTotalSell += $totSellPrice;
            $costPrice     = ($indvAvgPrice*$indvShare);
            $allTotalBuy  += $costPrice;
            $indGl         = ($totSellPrice-$costPrice);
            $indGLTotal   += $indGl;

            if($indGl>0){
                $gainLossWithCom	= ($indGl-$bCom);
            } else {
                $gainLossWithCom	= ($indGl-$bCom);
            }

            $allGainLoss 			+= $gainLossWithCom;
        }

        $realizeData = array(
            'allTotalBuy' 				=> $allTotalBuy,
            'allTotalSell' 				=> $allTotalSell,
            'indGLTotal' 				=> $indGLTotal,
            'allGainLoss' 				=> $allGainLoss,
        );

        return (object) $realizeData;
    }

    private function getYearlyChargeForWithdraw($portfolioCode, $toDate)
    {
        $yearlyCharge = 0;
        $yearlyChargeSql = "
            SELECT 
                YC.YEARLY_CHARGE AS YEARLY_CHARGE
            FROM 
                IS_YEARLY_CHARGE YC
            WHERE 
                YC.PORTFOLIO_CODE 	= :portfolio_code
            AND YC.CHARGE_YEAR 		<= :to_date
        ";

        $yearlyChargeSqlResult = DB::select(
            $yearlyChargeSql,
            [
                'to_date'        => $toDate->format('Y'),
                'portfolio_code' => $portfolioCode,
            ]
        );

        $yearlyCharge = collect($yearlyChargeSqlResult)->sum('yearly_charge');

        if ($yearlyCharge <= 0) {
            $yearlyChargeSql = "
                SELECT 
					INVCHARGE.CHARGE_VALUE 	AS CHARGE_VALUE
				FROM 
					IS_INVESTOR_CHARGE  	INVCHARGE,
					LS_TEMPLATE_CHARGE  	TEMPCHARGE,
					LS_PORTFOLIO_CHARGE 	PTCHARGE,
					GS_CHARGE_LIST      	CHARGELIST
				WHERE 
                    INVCHARGE.TEMPLATE_CHARGE_ID 	= TEMPCHARGE.TEMPLATE_CHARGE_ID
                    AND 	TEMPCHARGE.PORTFOLIO_CHARGE_ID 	= PTCHARGE.PORTFOLIO_CHARGE_ID
                    AND 	PTCHARGE.CHARGE_LIST_ID 		= CHARGELIST.CHARGE_LIST_ID
                    AND 	CHARGELIST.CHARGE_CODE 			= 8
                    AND 	INVCHARGE.END_DATE IS NULL
                    AND 	INVCHARGE.PORTFOLIO_CODE 		= :portfolio_code
            ";

            $yearlyChargeSqlResult = DB::select(
                $yearlyChargeSql,
                [
                    'portfolio_code' => $portfolioCode,
                ]
            );

            $yearlyCharge = collect($yearlyChargeSqlResult)->sum('charge_value');
        }

        return $yearlyCharge;
    }

    private function getEquityData($portfolioCode, $toDate)
    {
        $equitySql = "
            SELECT 
                NVL(PPST.CURRENT_BALANCE, 0) AS CURRENT_BALANCE, 
                PPST.EQUITY AS EQUITY, 
                PPST.ACCRUED_FEES_CHARGE, 
                NVL(PPST.REALIZE_GAIN_LOSS, 0) AS REALIZED_GAINLOSS, 
                NVL(CSTVAL.TOTAL_COST_AMOUNT, 0) AS COST_VALUE_SECURITIES, 
                NVL(CSTVAL.TOTAL_MKT_AMOUNT, 0) AS MKT_VALUE_SECURITIES, 
                NVL(
                    (PPST.COST_VALUE_SECURITIES), 
                    0
                ) AS COST_VALUE_SECURITIES_MAIN 
            FROM 
            P_PORTFOLIO_STATEMENT_TEMP PPST, 
            (
                SELECT 
                    ISCMS.PORTFOLIO_CODE, 
                    SUM(
                        NVL (
                        ISCMS.FRACTION_TOTAL_SHARE * ISCMS.NEW_AVERAGE_RATE, 
                        0
                        )
                    ) AS TOTAL_COST_AMOUNT, 
                    SUM(
                        NVL (
                        P_CAP_MKT_PRICE.CLOSE_PRICE * ISCMS.FRACTION_TOTAL_SHARE, 
                        0
                        )
                    ) AS TOTAL_MKT_AMOUNT 
                FROM 
                    IS_CAP_MKT_STOCK ISCMS, 
                    LS_INSTRUMENT_DETAILS LSID 
                LEFT JOIN (
                    SELECT 
                        CMPH.INSTRUMENT_DETAILS_ID, 
                        CMPH.CLOSE_PRICE 
                    FROM 
                        P_CAP_MKT_PRICE_HISTORY CMPH, 
                        LS_INSTRUMENT_DETAILS INSTDET 
                    WHERE 
                        INSTDET.INSTRUMENT_DETAILS_ID = CMPH.INSTRUMENT_DETAILS_ID 
                        AND CMPH.BUSINESS_DATE = (
                            SELECT 
                                MAX (
                                    P_CAP_MKT_PRICE_HISTORY.BUSINESS_DATE
                                ) 
                            FROM 
                                P_CAP_MKT_PRICE_HISTORY 
                            WHERE 
                                P_CAP_MKT_PRICE_HISTORY.BUSINESS_DATE <= :to_date
                                AND INSTDET.INSTRUMENT_DETAILS_ID = P_CAP_MKT_PRICE_HISTORY.INSTRUMENT_DETAILS_ID
                        )
                ) P_CAP_MKT_PRICE ON P_CAP_MKT_PRICE.INSTRUMENT_DETAILS_ID = LSID.INSTRUMENT_DETAILS_ID 
                WHERE 
                    ISCMS.CAP_MKT_STOCK_ID = (
                        SELECT 
                            MAX (ISCMSD.CAP_MKT_STOCK_ID) 
                        FROM 
                            IS_CAP_MKT_STOCK ISCMSD 
                        WHERE 
                            ISCMSD.INSTRUMENT_DETAILS_ID = ISCMS.INSTRUMENT_DETAILS_ID 
                            AND ISCMS.PORTFOLIO_CODE = ISCMSD.PORTFOLIO_CODE 
                            AND LSID.INSTRUMENT_DETAILS_ID = ISCMSD.INSTRUMENT_DETAILS_ID 
                            AND ISCMSD.BUSINESS_DATE <= :to_date
                    ) 
                    AND ISCMS.PORTFOLIO_CODE = :portfolio_code
                    AND ISCMS.FRACTION_TOTAL_SHARE > 0 
                GROUP BY 
                    ISCMS.PORTFOLIO_CODE
            ) CSTVAL 
            WHERE 
                PPST.PORTFOLIO_CODE = :portfolio_code
                AND PPST.PORTFOLIO_CODE = CSTVAL.PORTFOLIO_CODE 
                AND PPST.BUSINESS_DATE = (
                    SELECT 
                        MAX(
                            P_PORTFOLIO_STATEMENT_TEMP.BUSINESS_DATE
                        ) 
                    FROM 
                        P_PORTFOLIO_STATEMENT_TEMP P_PORTFOLIO_STATEMENT_TEMP, 
                        L_BUSINESS_DATE BDAT 
                    WHERE 
                        P_PORTFOLIO_STATEMENT_TEMP.PORTFOLIO_CODE = PPST.PORTFOLIO_CODE 
                        AND P_PORTFOLIO_STATEMENT_TEMP.BUSINESS_DATE = BDAT.BUSINESS_DATE 
                        AND PPST.PORTFOLIO_CODE = :portfolio_code
                        AND P_PORTFOLIO_STATEMENT_TEMP.BUSINESS_DATE <= :to_date
                )
        ";

        $equitySqlResult = DB::select(
            $equitySql,
            [
                'to_date'        => $toDate->format('Y-m-d'),
                'portfolio_code' => $portfolioCode,
            ]
        );

        $equitySqlResultCollection = collect($equitySqlResult)->first();

        return $equitySqlResultCollection;
    }
    
    private function getClosingPortData(string $portfolioCode, Carbon $toDate, Carbon $compareDate)
    {

        if ($compareDate->gte($toDate)) {
            $equitySql = "
                SELECT 
                    NVL(PPST.CURRENT_BALANCE, 0) AS CURRENT_BALANCE, 
                    PPST.EQUITY AS EQUITY, 
                    PPST.ACCRUED_FEES_CHARGE AS ACCRUED, 
                    NVL(PPST.REALIZE_GAIN_LOSS, 0) AS REALIZED_GAINLOSS, 
                    NVL(PPST.MARKET_VALUE_SECURITIES, 0) AS MKT_VALUE_SEC, 
                    NVL(
                        (PPST.COST_VALUE_SECURITIES), 
                        0
                    ) AS COST_VALUE_SECURITIES 
                    FROM 
                    P_PORTFOLIO_STATEMENT_TEMP PPST 
                    WHERE 
                    PPST.PORTFOLIO_CODE = :portfolio_code
                    AND PPST.BUSINESS_DATE = (
                        SELECT 
                        MAX(
                            P_PORTFOLIO_STATEMENT_TEMP.BUSINESS_DATE
                        ) 
                        FROM 
                        P_PORTFOLIO_STATEMENT_TEMP P_PORTFOLIO_STATEMENT_TEMP, 
                        L_BUSINESS_DATE BDAT 
                        WHERE 
                        P_PORTFOLIO_STATEMENT_TEMP.PORTFOLIO_CODE = PPST.PORTFOLIO_CODE 
                        AND P_PORTFOLIO_STATEMENT_TEMP.BUSINESS_DATE = BDAT.BUSINESS_DATE 
                        AND PPST.PORTFOLIO_CODE = :portfolio_code
                        AND P_PORTFOLIO_STATEMENT_TEMP.BUSINESS_DATE <= :to_date
                    )


            ";
        } else {
            $equitySql = "
                SELECT NVL(PPST.CURRENT_BALANCE,0) AS CURRENT_BALANCE,
                    PPST.EQUITY AS EQUITY,
                    NVL(PPST.ACCRUED_FEES_CHARGE,0) AS ACCRUED,
                    NVL(PPST.REALIZE_GAIN_LOSS,0) 	AS REALIZED_GAINLOSS,
                    NVL((PPST.MARKET_VALUE_SECURITIES), 0) 	AS MKT_VALUE_SEC,
                    NVL((PPST.COST_VALUE_SECURITIES), 0) 	AS COST_VALUE_SECURITIES
                FROM P_INVESTOR_PORTFOLIO_STATEMENT PPST
                    WHERE PPST.PORTFOLIO_CODE = :portfolio_code
                    AND PPST.BUSINESS_DATE =
                                    (SELECT MAX(P_PORTFOLIO_STATEMENT_TEMP.BUSINESS_DATE)
                                        FROM P_INVESTOR_PORTFOLIO_STATEMENT P_PORTFOLIO_STATEMENT_TEMP, L_BUSINESS_DATE BDAT
                                        WHERE P_PORTFOLIO_STATEMENT_TEMP.PORTFOLIO_CODE = PPST.PORTFOLIO_CODE
                                        AND P_PORTFOLIO_STATEMENT_TEMP.BUSINESS_DATE = BDAT.BUSINESS_DATE
                                        AND PPST.PORTFOLIO_CODE = :portfolio_code
                                        AND P_PORTFOLIO_STATEMENT_TEMP.BUSINESS_DATE <= :to_date
                                        )
            ";
        }
        

        $equitySqlResult = DB::select(
            $equitySql,
            [
                'to_date'        => $toDate->format('Y-m-d'),
                'portfolio_code' => $portfolioCode,
            ]
        );

        $equitySqlResultCollection = collect($equitySqlResult)->first();

        return $equitySqlResultCollection;
    }
}
