<?php

namespace App\Http\Controllers\Investor;

use Carbon\Carbon;
use App\Models\BusinessDate;
use Illuminate\Http\Request;
use App\Models\InvestorDetail;
use App\Http\Controllers\Controller;
use App\Models\InvestorChangeRequest;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class InvestorInformationController extends Controller
{
    public function personalInformation()
    {
        $user = Auth::user();
        $investorDetail = InvestorDetail::where('portfolio_code', $user->investor_code)->with(['templateDetails', 'branchDetails'],)->first();

        // dd($investorDetail->branchDetails->bankDetails);
        
        
        return response()->json([
            'status'   => 'success',
            'investor' => [
                'investor_name'     => $investorDetail->investor_name,
                'father_name'       => $investorDetail->father_name,
                'mother_name'       => $investorDetail->mother_name,
                'residency'         => $investorDetail->residency,
                'portfolio_code'    => $investorDetail->portfolio_code,
                'mailing_address'   => $investorDetail->mailing_address,
                'mobile'            => $investorDetail->mobile,
                'email'             => $investorDetail->email,
                'nid'               => $investorDetail->nid,
                'bank_account_no'   => $investorDetail->bank_account_no,
                'bo_account_no'     => $investorDetail->bo_account_no,
                'bank_id'           => $investorDetail->branchDetails->bankDetails->org_id,
                'bank_name'         => $investorDetail->branchDetails->bankDetails->org_name,
                'branch_id'         => $investorDetail->branchDetails->org_branch_id,
                'branch_name'       => $investorDetail->branchDetails->branch_name,
                'branch_routing_no' => $investorDetail->branchDetails->routing_no,
                'product_name'      => $investorDetail->templateDetails->product_name,
                'dob'               => empty($investorDetail->dob) || is_null($investorDetail->dob) ? 'N/A' : Carbon::createFromFormat( 'Y-m-d H:i:s', $investorDetail->dob)->format('d-M-Y'),
            ]
        ]);
    }

    public function submitChangeRequest(Request $request)
    {
        $validationLists = [];
        $validationRules = [];
        $fieldLists = [
            'name',
            'mailing_address',
            'nid',
            // 'nid_photo',
            'tin',
            // 'tin_photo',
            'bank_id',
            'branch_id',
            'bank_account_no',
            'bank_leaf',
        ];

        $validator = Validator::make(
            $request->all(),
            [
                'change_field' => 'required',
            ],
            [
                'change_field.required' => 'This field must be required as marked to change corresponding input value',
            ]
        );

        if ($validator->fails()) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Form validation error',
                'errors' => $validator->errors()->messages(),
            ], 400);
        }

        $validationLists[] = $fieldLists[$request->change_field];

        foreach ($validationLists as $key => $fieldName) {
            if (in_array($fieldName, ['bank_id', 'branch_id', 'bank_account_no', 'bank_leaf']) && !in_array($fieldName, $validationRules)) {
                $validationRules['bank_id'] = 'required';
                $validationRules['branch_id'] = 'required';
                $validationRules['bank_account_no'] = 'required';
                $validationRules['bank_leaf'] = 'required|file|extensions:jpg,png,jpeg,pdf|max:5120';
            } else {
                $validationRules[$fieldName] = 'required';
                if($request->nid){
                    $validationRules['nid'] = ['required', 'regex:/^(?:\d{10}|\d{17})$/'];
                    $validationRules['nid_photo'] = 'required|file|extensions:jpg,png,jpeg,pdf|max:5120';
                }else if($request->tin){
                    $validationRules['tin_photo'] = 'required|file|extensions:jpg,png,jpeg,pdf|max:5120';
                }
            }

        }

        $validator = Validator::make(
            $request->all(),
            $validationRules,
            [
                'name.required'            => 'The name filed is required',
                'mailing_address.required' => 'The mailing address filed is required',
                
                'nid.required'         => 'The nid filed is required',
                'nid.regex'           => 'The NID must be either 10 or 17 digits long.',
                'nid_photo.file'       => 'The nid filed is required',
                'nid_photo.extensions' => 'The nid photo must be in jpg, png, jpeg or pdf format',
                'nid_photo.max'        => 'The nid photo must be less than 5 mb',

                'tin.required'         => 'The tin filed is required',
                'tin_photo.file'       => 'The tin filed is required',
                'tin_photo.extensions' => 'The tin photo must be in jpg, png, jpeg or pdf format',
                'tin_photo.max'        => 'The tin photo must be less than 5 mb',

                
                'bank_id.required'         => 'The bank filed is required',
                'branch_id.required'       => 'The branch filed is required',
                'bank_account_no.required' => 'The bank account number filed is required',
                'bank_leaf.required'       => 'The bank leaf file is required',
                'bank_leaf.file'           => 'The bank leaf must be a file',
                'bank_leaf.max'            => 'The bank leaf must be less than 5 mb',
                'bank_leaf.extensions'     => 'The bank leaf must be in jpg, png, jpeg or pdf format',
            ]
        );

        if ($validator->fails()) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Form validation error',
                'errors' => $validator->errors()->messages(),
            ], 400);
        } else {
            try {
                $bankLeadFileName = null;
                $user                 = Auth::user();
                $businessDate         = BusinessDate::latest('business_date')->first();

                if ($request->hasFile('bank_leaf')) {
                    $bankLeadFile     = $request->file('bank_leaf');
                    $bankLeadFileName = 'bank_leaf_'.$user->investor_code.'_'.time().'.'.$bankLeadFile->getClientOriginalExtension();
                    $bankLeadFile->move(public_path().'/upload/bank_leaf', $bankLeadFileName);    
                }
                $nidFileName = null;
                if ($request->hasFile('nid_photo')) {
                    $nidFile     = $request->file('nid_photo');
                    $nidFileName = 'nid_photo_'.$user->investor_code.'_'.time().'.'.$nidFile->getClientOriginalExtension();
                    $nidFile->move(public_path().'/upload/nid_photo', $nidFileName);    
                }

                $tinFileName=null;
                if ($request->hasFile('tin_photo')) {
                    $tinFile     = $request->file('tin_photo');
                    $tinFileName = 'tin_photo_'.$user->investor_code.'_'.time().'.'.$tinFile->getClientOriginalExtension();
                    $tinFile->move(public_path().'/upload/tin_photo', $tinFileName);    
                }


                $investorChangeRequest                      = new InvestorChangeRequest();
                $investorChangeRequest->investor_details_id = $user->investorDetail->investor_details_id;
                $investorChangeRequest->name                = $request->name;
                $investorChangeRequest->mailing_address     = $request->mailing_address;

                $investorChangeRequest->nid                 = $request->nid;
                $investorChangeRequest->nid_photo           = $nidFileName;

                $investorChangeRequest->tin                 = $request->tin;
                $investorChangeRequest->tin_photo           = $tinFileName;


                $investorChangeRequest->bank_id             = $request->bank_id;
                $investorChangeRequest->branch_id           = $request->branch_id;
                $investorChangeRequest->account_no          = $request->bank_account_no;
                $investorChangeRequest->bank_leaf           = $bankLeadFileName;
                $investorChangeRequest->status              = 'pending';
                $investorChangeRequest->business_date       = $businessDate->business_date;
                $investorChangeRequest->record_date         = now();

                if ($investorChangeRequest->save()) {
                    return response()->json([
                        'status'      => 'success',
                        'message'   => 'Change request submitted successfully',
                    ]);
                }
            } catch (\Throwable $th) {
                return response()->json([
                    'status'      => 'error',
                    'dev_message' => $th->getMessage(),
                    'message'      => 'Change request form submit unsuccessful',
                ], 400);
            }
        }
    }
}
