<?php

namespace App\Http\Controllers\Report;

use Carbon\Carbon;
use App\Models\BusinessDate;
use Illuminate\Http\Request;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class PortfolioStatementController extends Controller
{
	public function portfolioStatementDownload(Request $request)
	{
		$validator = Validator::make(
			$request->all(),
			[
				'as_on_date'  => 'required|date_format:Y-m-d',
			],
			[
				'as_on_date.required'          => 'The as on date filed is required',
				'as_on_date.date_format'       => 'The as on date field must be in Y-m-d format',
			]
		);

		if ($validator->fails()) {
			return response()->json([
				'status'  => 'error',
				'message' => 'Form validation error',
				'errors' => $validator->errors()->messages(),
			], 400);
		}

		$data = [
			'title' => 'Portfolio Statement (Date Wise)',
		];

		$user                    = Auth::user();
		$investorDetail          = $user->investorDetail;
		$businessDate            = BusinessDate::latest('business_date')->first();
		$requestDate             = Carbon::createFromFormat('Y-m-d', $request->as_on_date);
		$data['account_type']    = $investorDetail->templateDetails->product_short_name;
		$data['account_no']      = $investorDetail->portfolio_code;
		$data['bo_no']           = $investorDetail->bo_account_no;
		$data['account_date']    = Carbon::createFromFormat('Y-m-d H:i:s', $investorDetail->business_date)->format(env('OUTPUT_DATE_FORMAT'));
		$data['account_name']    = $investorDetail->investor_name;
		$data['account_address'] = $investorDetail->mailing_address;
		$data['report_date']     = Carbon::createFromFormat('Y-m-d', $requestDate->format('Y-m-d'))->format(env('OUTPUT_DATE_FORMAT'));

		if ($investorDetail->account_enable_disable == 'e') {
			$data['account_status'] = 'Active';
		}

		if ($investorDetail->account_close == 'y') {
			$data['account_status'] = 'Close';
		}

		if ($investorDetail->account_suspend == 'y') {
			$data['account_status'] = 'Suspend';
		}

		$check_template = $this->getTemplate($user->investor_code);
		if ($check_template) {
			$template		= $check_template[0]->is_template;
			$template_id 	= $check_template[0]->template_id;
		} else {
			$template 		= 'n';
			$template_id	= '';
		}

		$stocks = $this->getStock($user->investor_code,$template,$template_id,$requestDate);
		$data['marginable_securities']     = collect($stocks)->where('margin_or_nonmargin_flag', 'y')->where('total_share', '>', 0)->values()->all();
		$data['non_marginable_securities'] = collect($stocks)->where('margin_or_nonmargin_flag', 'n')->where('total_share', '>', 0)->values()->all();
		$data['accountStatus'] = $this->getPortfolioStatementData($requestDate,$user->investor_code);

		$getCashDivident = $this->getCashDivident($user->investor_code, $requestDate);
		$getstockDivident = $this->getStockDivident($user->investor_code, $requestDate);
		$getIPOInfo = $this->getIpoInformation($user->investor_code, $requestDate);
		
		$getUnClearChequeTotAmount = $this->getUnClearChequeTotAmount($user->investor_code, $requestDate);
		if($getUnClearChequeTotAmount){
			$data['unclearCheque'] = $getUnClearChequeTotAmount[0]->totalunclearchq;
		}else{
			$data['unclearCheque'] = 0;
		}

		$getRightShareApplliedTotAmount = $this->getRightShareApplliedTotAmount($user->investor_code, $requestDate);
		if($getRightShareApplliedTotAmount){
			$data['total_right_share'] = $getRightShareApplliedTotAmount[0]->totalrsapply;
		}else{
			$data['total_right_share'] = 0;
		}

		$getIPOApplliedTotAmount = $this->getIPOApplliedTotAmount($user->investor_code, $requestDate);
		if($getIPOApplliedTotAmount){
			$data['totalIPOApplicationAmount'] = $getIPOApplliedTotAmount[0]->totalipoapply;
		}else{
			$data['totalIPOApplicationAmount'] = 0;
		}

		$data['requestDate']               = $requestDate;
		$data['allCostAmount']             = collect($stocks)->where('total_share', '>', 0)->sum('total_cost_amount');
		$data['cash_dividend'] = collect($getCashDivident);
		$data['ipo_information'] = collect($getIPOInfo);
		$data['stock_dividend'] = collect(collect($getstockDivident)->where('stock_div', '>', 0)->values()->all());

		$reportName = 'PortfolioStatement_' . $requestDate->format('d_m_Y') . '.pdf';
		$pdf = Pdf::loadView('portfolio.index', $data);
		$pdf->setPaper('a4');
		return $pdf->stream($reportName);
	}

	private function getTemplate($portfolioCode)
	{

		$sql = "SELECT 
					IS_TEMPLATE,
					TEMPLATE_ID
			  	FROM  
					IS_INVESTOR_DETAILS
				WHERE 
					PORTFOLIO_CODE=:portfolio_code";

		$sql = DB::select(
			$sql,
			[
				'portfolio_code' => $portfolioCode
			]
		);

		return $sql;
	}

	private function getStock($portfolioCode,$template,$template_id,$requestDate){
		if ($template == 'y') {
			$sql = "
			SELECT 
				DISTINCT ISCMS.INSTRUMENT_DETAILS_ID, 
				INDV.IS_MARGIN_LOAN_ACTIVE, 
				INDV.ACCOUNT_CLOSE, 
				INDV.ACCOUNT_SUSPEND, 
				INDV.ACCOUNT_ENABLE_DISABLE, 
				SUM(ISCMS.TOTAL_SHARE) AS TOTAL_SHARE, 
				(
					SUM(
					ISCMS.TOTAL_SHARE * ISCMS.AVG_RATE
					)/ SUM(ISCMS.TOTAL_SHARE)
				) AS AVG_RATE, 
				SUM(
					ISCMS.TOTAL_SHARE * ISCMS.AVG_RATE
				) AS TOTAL_COST_AMOUNT, 
				SUM(
					P_CAP_MKT_PRICE.CLOSE_PRICE * ISCMS.TOTAL_SHARE
				) AS TOTAL_MKT_AMOUNT, 
				SUM(ISCMS.MATURED_SHARE) AS MATURED_SHARE, 
				INDV.IS_TEMPLATE, 
				LSID.INSTRUMENT_NAME, 
				P_CAP_MKT_PRICE.CLOSE_PRICE, 
				IBRKCHG.CHARGE, 
				GS_SECTOR.SECTOR_NAME, 
				(
					SELECT 
					GS_SATTLEMENT_CAT.SATTLEMENT_CAT_NAME 
					FROM 
					GS_SATTLEMENT_CAT 
					WHERE 
					GS_SATTLEMENT_CAT.SATTLEMENT_CAT_ID = LSID.SATTLEMENT_CAT_ID
				) AS SATTLEMENT_CAT_NAME, 
				(
					SELECT 
					DISTINCT LS_MARKET_WISE_INSTRUMENT.SYMBOL 
					FROM 
					LS_MARKET_WISE_INSTRUMENT 
					WHERE 
					LS_MARKET_WISE_INSTRUMENT.INSTRUMENT_DETAILS_ID = LSID.INSTRUMENT_DETAILS_ID
				) AS SYMBOL, 
				(
					SELECT 
					DISTINCT LS_MARKET_WISE_INSTRUMENT.MARGIN_OR_NONMARGIN_FLAG 
					FROM 
					LS_MARKET_WISE_INSTRUMENT 
					WHERE 
					LS_MARKET_WISE_INSTRUMENT.INSTRUMENT_DETAILS_ID = LSID.INSTRUMENT_DETAILS_ID 
					AND LS_MARKET_WISE_INSTRUMENT.MARKET_DETAILS_ID = 61
				) AS MARGIN_OR_NONMARGIN_FLAG, 
				NVL(
					(
					SELECT 
						P_CAP_MKT_PRICE_HISTORY.CLOSE_PRICE 
					FROM 
						P_CAP_MKT_PRICE_HISTORY 
					WHERE 
						P_CAP_MKT_PRICE_HISTORY.INSTRUMENT_DETAILS_ID = LSID.INSTRUMENT_DETAILS_ID 
						AND P_CAP_MKT_PRICE_HISTORY.CAP_MKT_PRICE_HISTORY_ID = (
						SELECT 
							MAX(
							P_CAP_MKT_PRICE_HISTORY.CAP_MKT_PRICE_HISTORY_ID
							) 
						FROM 
							P_CAP_MKT_PRICE_HISTORY 
						WHERE 
							P_CAP_MKT_PRICE_HISTORY.INSTRUMENT_DETAILS_ID = LSID.INSTRUMENT_DETAILS_ID 
							AND P_CAP_MKT_PRICE_HISTORY.BUSINESS_DATE < (
							SELECT 
								MAX(
								P_CAP_MKT_PRICE_HISTORY.BUSINESS_DATE
								) 
							FROM 
								P_CAP_MKT_PRICE_HISTORY 
							WHERE 
								P_CAP_MKT_PRICE_HISTORY.INSTRUMENT_DETAILS_ID = LSID.INSTRUMENT_DETAILS_ID 
								AND P_CAP_MKT_PRICE_HISTORY.BUSINESS_DATE <= :request_date
							)
						)
					), 
					0
				) AS YCP, 
				NVL(
					(
					SELECT 
						P_PE_RATIO.RATIO_VALUE 
					FROM 
						P_PE_RATIO 
					WHERE 
						P_PE_RATIO.INSTRUMENT_DETAILS_ID = LSID.INSTRUMENT_DETAILS_ID 
						AND P_PE_RATIO.P_PE_RATIO_ID = (
						SELECT 
							MAX(P_PE_RATIO.P_PE_RATIO_ID) 
						FROM 
							P_PE_RATIO 
						WHERE 
							P_PE_RATIO.INSTRUMENT_DETAILS_ID = LSID.INSTRUMENT_DETAILS_ID 
							AND P_PE_RATIO.BUSINESS_DATE < (
							SELECT 
								MAX(P_PE_RATIO.BUSINESS_DATE) 
							FROM 
								P_PE_RATIO 
							WHERE 
								P_PE_RATIO.INSTRUMENT_DETAILS_ID = LSID.INSTRUMENT_DETAILS_ID 
								AND P_PE_RATIO.BUSINESS_DATE <= :request_date
							)
						)
					), 
					0
				) AS RATIO_VALUE 
				FROM 
					IS_CAP_MKT_STOCK ISCMS, 
					LS_INSTRUMENT_DETAILS LSID, 
					P_CAP_MKT_PRICE_HISTORY P_CAP_MKT_PRICE, 
					IS_I_BROKER_CHARGE IBRKCHG, 
					IS_INVESTOR_DETAILS INDV, 
					GS_SECTOR 
				WHERE 
				ISCMS.CAP_MKT_STOCK_ID = (
					SELECT 
					MAX(ISCMSD.CAP_MKT_STOCK_ID) 
					FROM 
					IS_CAP_MKT_STOCK ISCMSD 
					WHERE 
					ISCMSD.INSTRUMENT_DETAILS_ID = ISCMS.INSTRUMENT_DETAILS_ID 
					AND ISCMS.PORTFOLIO_CODE = ISCMSD.PORTFOLIO_CODE 
					AND LSID.INSTRUMENT_DETAILS_ID = ISCMSD.INSTRUMENT_DETAILS_ID 
					AND P_CAP_MKT_PRICE.INSTRUMENT_DETAILS_ID = ISCMSD.INSTRUMENT_DETAILS_ID 
					AND ISCMSD.PORTFOLIO_CODE IN (
						SELECT 
						PORTFOLIO_CODE 
						FROM 
						IS_INVESTOR_DETAILS 
						WHERE 
						IS_INVESTOR_DETAILS.TEMPLATE_ID = :template_id 
						AND IS_TEMPLATE = :template
					) 
					AND ISCMSD.BUSINESS_DATE <= :request_date
				) 
				AND P_CAP_MKT_PRICE.INSTRUMENT_DETAILS_ID = ISCMS.INSTRUMENT_DETAILS_ID 
				AND P_CAP_MKT_PRICE.INSTRUMENT_DETAILS_ID = LSID.INSTRUMENT_DETAILS_ID 
				AND ISCMS.I_BROKER_CHARGE_ID = IBRKCHG.I_BROKER_CHARGE_ID 
				AND INDV.PORTFOLIO_CODE = ISCMS.PORTFOLIO_CODE 
				AND ISCMS.PORTFOLIO_CODE = IBRKCHG.PORTFOLIO_CODE 
				AND GS_SECTOR.SECTOR_ID = LSID.SECTOR_ID 
				AND ISCMS.TOTAL_SHARE > 0 
				AND P_CAP_MKT_PRICE.CAP_MKT_PRICE_HISTORY_ID = (
					SELECT 
					MAX(
						P_CAP_MKT_PRICE_HISTORY.CAP_MKT_PRICE_HISTORY_ID
					) 
					FROM 
					P_CAP_MKT_PRICE_HISTORY 
					WHERE 
					P_CAP_MKT_PRICE_HISTORY.INSTRUMENT_DETAILS_ID = LSID.INSTRUMENT_DETAILS_ID 
					AND P_CAP_MKT_PRICE_HISTORY.BUSINESS_DATE = (
						SELECT 
						MAX(
							P_CAP_MKT_PRICE_HISTORY.BUSINESS_DATE
						) 
						FROM 
						P_CAP_MKT_PRICE_HISTORY 
						WHERE 
						P_CAP_MKT_PRICE_HISTORY.INSTRUMENT_DETAILS_ID = LSID.INSTRUMENT_DETAILS_ID 
						AND P_CAP_MKT_PRICE_HISTORY.BUSINESS_DATE <= :request_date
					)
				) 
				GROUP BY 
				ISCMS.INSTRUMENT_DETAILS_ID, 
				INDV.IS_MARGIN_LOAN_ACTIVE, 
				INDV.ACCOUNT_CLOSE, 
				INDV.ACCOUNT_SUSPEND, 
				INDV.ACCOUNT_ENABLE_DISABLE, 
				INDV.IS_TEMPLATE, 
				LSID.INSTRUMENT_NAME, 
				P_CAP_MKT_PRICE.CLOSE_PRICE, 
				IBRKCHG.CHARGE, 
				GS_SECTOR.SECTOR_NAME, 
				LSID.INSTRUMENT_DETAILS_ID, 
				LSID.SATTLEMENT_CAT_ID 
				ORDER BY 
				MARGIN_OR_NONMARGIN_FLAG DESC, 
				SYMBOL, 
				PORTFOLIO_CODE ASC
			";
			$sql = DB::select(
				$sql,
				[
					'request_date' => $requestDate->format('Y-m-d'),
					'template' => $template,
					'template_id' => $template_id
				]
			);
		} else {
			$sql = "
			SELECT 
				STOCK_INFO.INSTRUMENT_DETAILS_ID AS INSTRUMENT_DETAILS_ID, 
				INDV.PORTFOLIO_CODE AS PORTFOLIO_CODE, 
				INDV.IS_MARGIN_LOAN_ACTIVE AS IS_MARGIN_LOAN_ACTIVE, 
				INDV.ACCOUNT_CLOSE AS ACCOUNT_CLOSE, 
				INDV.ACCOUNT_SUSPEND AS ACCOUNT_SUSPEND, 
				INDV.ACCOUNT_ENABLE_DISABLE AS ACCOUNT_ENABLE_DISABLE, 
				COALESCE (
					INDV.INSTITUTION_NAME, INDV.INVESTOR_NAME
				) AS INVESTOR_NAME, 
				STOCK_INFO.TOTAL_SHARE AS TOTAL_SHARE, 
				STOCK_INFO.AVG_RATE AS AVG_RATE, 
				STOCK_INFO.TOTAL_COST_AMOUNT AS TOTAL_COST_AMOUNT, 
				STOCK_INFO.TOTAL_MKT_AMOUNT AS TOTAL_MKT_AMOUNT, 
				STOCK_INFO.MATURED_SHARE AS MATURED_SHARE, 
				INDV.IS_TEMPLATE AS IS_TEMPLATE, 
				STOCK_INFO.INSTRUMENT_NAME AS INSTRUMENT_NAME, 
				NVL (STOCK_INFO.CLOSE_PRICE, 0.1) AS CLOSE_PRICE, 
				STOCK_INFO.CHARGE AS CHARGE, 
				STOCK_INFO.SECTOR_NAME AS SECTOR_NAME, 
				STOCK_INFO.SATTLEMENT_CAT_NAME AS SATTLEMENT_CAT_NAME, 
				STOCK_INFO.SYMBOL AS SYMBOL, 
				STOCK_INFO.MARGIN_OR_NONMARGIN_FLAG AS MARGIN_OR_NONMARGIN_FLAG, 
				STOCK_INFO.YCP AS YCP, 
				STOCK_INFO.RATIO_VALUE AS RATIO_VALUE 
			FROM 
				IS_INVESTOR_DETAILS INDV 
			LEFT JOIN (
					SELECT 
					ISCMS.PORTFOLIO_CODE, 
					ISCMS.INSTRUMENT_DETAILS_ID AS INSTRUMENT_DETAILS_ID, 
					ISCMS.TOTAL_SHARE AS TOTAL_SHARE, 
					ISCMS.AVG_RATE AS AVG_RATE, 
					NVL (
						ISCMS.TOTAL_SHARE * ISCMS.AVG_RATE, 
						0
					) AS TOTAL_COST_AMOUNT, 
					NVL (
						P_CAP_MKT_PRICE.CLOSE_PRICE * ISCMS.TOTAL_SHARE, 
						0
					) AS TOTAL_MKT_AMOUNT, 
					ISCMS.MATURED_SHARE AS MATURED_SHARE, 
					LSID.INSTRUMENT_NAME AS INSTRUMENT_NAME, 
					P_CAP_MKT_PRICE.CLOSE_PRICE AS CLOSE_PRICE, 
					0 AS CHARGE, 
					GSECT.SECTOR_NAME AS SECTOR_NAME, 
					SATCAT.SATTLEMENT_CAT_NAME AS SATTLEMENT_CAT_NAME, 
					MWINSTDET.SYMBOL AS SYMBOL, 
					MWINSTDET.MARGIN_OR_NONMARGIN_FLAG, 
					0 AS YCP, 
					PE_RATIO.RATIO_VALUE AS RATIO_VALUE 
					FROM 
					GS_SECTOR GSECT, 
					GS_SATTLEMENT_CAT SATCAT, 
					IS_CAP_MKT_STOCK ISCMS, 
					(
						SELECT 
						SYMBOL, 
						INSTRUMENT_DETAILS_ID, 
						MARGIN_OR_NONMARGIN_FLAG 
						FROM 
						LS_MARKET_WISE_INSTRUMENT 
						WHERE 
						MARKET_DETAILS_ID = 61
					) MWINSTDET, 
					LS_INSTRUMENT_DETAILS LSID 
					LEFT JOIN (
						SELECT 
						P_PE_RATIO.INSTRUMENT_DETAILS_ID, 
						P_PE_RATIO.RATIO_VALUE 
						FROM 
						P_PE_RATIO 
						WHERE 
						P_PE_RATIO.BUSINESS_DATE = (
							SELECT 
							MAX (P_PE_RATIO.BUSINESS_DATE) 
							FROM 
							P_PE_RATIO 
							WHERE 
							P_PE_RATIO.BUSINESS_DATE <= :request_date
						)
					) PE_RATIO ON PE_RATIO.INSTRUMENT_DETAILS_ID = LSID.INSTRUMENT_DETAILS_ID 
					LEFT JOIN (
						SELECT 
						CMPH.INSTRUMENT_DETAILS_ID, 
						CMPH.CLOSE_PRICE 
						FROM 
						P_CAP_MKT_PRICE_HISTORY CMPH, 
						LS_INSTRUMENT_DETAILS INSTDET 
						WHERE 
						INSTDET.INSTRUMENT_DETAILS_ID = CMPH.INSTRUMENT_DETAILS_ID 
						AND CMPH.BUSINESS_DATE = (
							SELECT 
							MAX (
								P_CAP_MKT_PRICE_HISTORY.BUSINESS_DATE
							) 
							FROM 
							P_CAP_MKT_PRICE_HISTORY 
							WHERE 
							P_CAP_MKT_PRICE_HISTORY.BUSINESS_DATE <= :request_date 
							AND INSTDET.INSTRUMENT_DETAILS_ID = P_CAP_MKT_PRICE_HISTORY.INSTRUMENT_DETAILS_ID
						)
					) P_CAP_MKT_PRICE ON P_CAP_MKT_PRICE.INSTRUMENT_DETAILS_ID = LSID.INSTRUMENT_DETAILS_ID 
					WHERE 
					ISCMS.CAP_MKT_STOCK_ID = (
						SELECT 
						MAX (ISCMSD.CAP_MKT_STOCK_ID) 
						FROM 
						IS_CAP_MKT_STOCK ISCMSD 
						WHERE 
						ISCMSD.INSTRUMENT_DETAILS_ID = ISCMS.INSTRUMENT_DETAILS_ID 
						AND ISCMS.PORTFOLIO_CODE = ISCMSD.PORTFOLIO_CODE 
						AND LSID.INSTRUMENT_DETAILS_ID = ISCMSD.INSTRUMENT_DETAILS_ID 
						AND ISCMSD.BUSINESS_DATE <= :request_date
					) 
					AND ISCMS.PORTFOLIO_CODE = :portfolio_code 
					AND ISCMS.TOTAL_SHARE > 0 
					AND MWINSTDET.INSTRUMENT_DETAILS_ID = LSID.INSTRUMENT_DETAILS_ID 
					AND GSECT.SECTOR_ID = LSID.SECTOR_ID 
					AND SATCAT.SATTLEMENT_CAT_ID = LSID.SATTLEMENT_CAT_ID
				) STOCK_INFO ON STOCK_INFO.PORTFOLIO_CODE = INDV.PORTFOLIO_CODE 
				LEFT JOIN LS_TEMPLATE LTEMP ON LTEMP.TEMPLATE_ID = INDV.TEMPLATE_ID 
				WHERE 
				INDV.PORTFOLIO_CODE = :portfolio_code 
				ORDER BY 
				INDV.PORTFOLIO_CODE, 
				MARGIN_OR_NONMARGIN_FLAG DESC, 
				SYMBOL ASC
			";
			$sql = DB::select(
				$sql,
				[
					'request_date' => $requestDate->format('Y-m-d'),
					'portfolio_code' =>$portfolioCode
				]
			);
		}
		return $sql;
	}

	private function getPortfolioStatementData($requestDate,$portfolioCode){
		$sql = "
			SELECT COALESCE (INVDET.INSTITUTION_NAME, INVDET.INVESTOR_NAME)
					   AS INVESTOR_NAME,
				   INVDET.PORTFOLIO_CODE,
				   INVDET.BO_ACCOUNT_NO,
				   INVDET.MOBILE,
				   NVL (INTEREST_DETAILS.INTEREST_RATE, 0)
					   AS INTEREST_RATE,
				   INVDET.EMAIL,
				   INVDET.TEMPLATE_ID,
				   INVDET.TENURE_YEAR,
				   INVDET.TENURE_MATURE_DATE,
				   INVDET.MONTHLY_SCHEME,
				   INVDET.BUSINESS_DATE
					   AS DATE_OF_ACC_OPENING,
				   INVDET.ACCOUNT_CLOSE,
				   INVDET.ACCOUNT_SUSPEND,
				   INVDET.ACCOUNT_ENABLE_DISABLE,
				   (CASE
						WHEN INVDET.IS_MARGIN_LOAN_ACTIVE = 'y' THEN 'Margin'
						ELSE 'Non Margin'
					END)
					   AS MARGIN_NONMARGIN,
				   NVL ((P_PORTFOLIO_STATEMENT_TEMP.PURCHASE_POWER), 0)
					   AS PURCHASE_POWER,
				   ''
					   AS PURCHASE_POWER_DR_CR,
				   NVL ((P_PORTFOLIO_STATEMENT_TEMP.RECEIVABLES_SALES), 0)
					   AS RECIEVABLE_SALES,
				   ''
					   AS RECIEVABLE_SALES_DR_CR,
				   NVL ((P_PORTFOLIO_STATEMENT_TEMP.TOTAL_DEPOSIT), 0)
					   AS TOTAL_DEPOSIT,
				   NVL ((REALIZED_GAINLOSS.REALIZED_GAIN_LOSS), 0)
					   AS REALIZED_GAINLOSS,
				   ''
					   AS REALIZED_GAINLOSS_DR_CR,
				   0
					   AS ROI,
				   ''
					   AS ROI_DR_CR,
				   NVL ((P_PORTFOLIO_STATEMENT_TEMP.DIVIDEND_INCOME), 0)
					   AS DIVIDEND_INCOME,
				   NVL ((P_PORTFOLIO_STATEMENT_TEMP.WITHDRAW), 0)
					   AS WITHDRAW,
				   NVL ((P_PORTFOLIO_STATEMENT_TEMP.MARKET_VALUE_SECURITIES), 0)
					   AS MKT_VALUE_SEC,
				   NVL ((P_PORTFOLIO_STATEMENT_TEMP.UNREALIZE_GAIN_LOSS), 0)
					   AS UNREALIZE_GL,
				   ''
					   AS UNREALIZE_GL_DR_CR,
				   NVL ((P_PORTFOLIO_STATEMENT_TEMP.EQUITY), 0)
					   AS EQUITY,
				   ''
					   AS EQUITY_DR_CR,
				   NVL ((P_PORTFOLIO_STATEMENT_TEMP.FUND_TRANSFER_IN), 0)
					   AS TRANSFERIN,
				   NVL ((P_PORTFOLIO_STATEMENT_TEMP.FUND_TRANSFER_OUT), 0)
					   AS TRANSFEROUT,
				   NVL ((P_PORTFOLIO_STATEMENT_TEMP.ACCRUED_FEES_CHARGE), 0)
					   AS ACCRUED,
				   NVL ((P_PORTFOLIO_STATEMENT_TEMP.EQUITY_TO_DEBT_RATIO), 0)
					   AS EQUITY_TO_DEBT_RATIO,
				   NVL ((P_PORTFOLIO_STATEMENT_TEMP.RECEIVABLE_DIVIDEND), 0)
					   AS CASH_DIV_RECEIVABLE,
				   NVL ((P_PORTFOLIO_STATEMENT_TEMP.RECEIVED_DIVIDEND), 0)
					   AS CASH_DIV_RECEIVED,
				   NVL ((P_PORTFOLIO_STATEMENT_TEMP.LOAN_RATIO), 0)
					   AS LOAN_RATIO,
				   NVL ((P_PORTFOLIO_STATEMENT_TEMP.ACCRUED_FEES_CHARGE), 0)
					   AS ACCRUED_FEES_CHARGE,
				   NVL ((P_PORTFOLIO_STATEMENT_TEMP.CURRENT_BALANCE), 0)
					   AS CURRENT_BALANCE,
				   SHARE_IN_DETAILS.TOTAL_SHARE_IN
					   AS SHARE_TRANSFER_IN,
				   SHARE_OUT_DETAILS.TOTAL_SHARE_OUT
					   AS SHARE_TRANSFER_OUT,
				   NVL ((PENDING_WITHDRAW.PAMOUNT), 0)
					   AS PAMOUNT,
				   NVL ((P_PORTFOLIO_STATEMENT_TEMP.IPO_ORDER_AMOUNT), 0)
					   AS IPO_ORDER_AMOUNT
			  FROM P_PORTFOLIO_STATEMENT_TEMP  P_PORTFOLIO_STATEMENT_TEMP,
				   IS_INVESTOR_DETAILS         INVDET
				   LEFT JOIN
				   (  SELECT X.PORTFOLIO_CODE, SUM (x.BROKER_COMMISSION) REALIZED_GAIN_LOSS
						FROM (  SELECT CM_STOCK.PORTFOLIO_CODE,
									   SUM (
										   (  (CM_STOCK.SHARES * CM_STOCK.ACTUAL_RATE)
											- (CM_STOCK.SHARES * CM_STOCK.AVG_RATE)
											- (CM_STOCK.BROKER_COMMISSION)))
										   AS BROKER_COMMISSION
								  FROM IS_CAP_MKT_STOCK CM_STOCK
								 WHERE     CM_STOCK.STATUS = 'enc'
									   AND CM_STOCK.STOCK_MATURE_DATE IS NULL
									   AND CM_STOCK.BUSINESS_DATE <=
										   :request_date
							  GROUP BY CM_STOCK.PORTFOLIO_CODE, CM_STOCK.BUSINESS_DATE) X
					GROUP BY X.PORTFOLIO_CODE) REALIZED_GAINLOSS
					   ON REALIZED_GAINLOSS.PORTFOLIO_CODE = INVDET.PORTFOLIO_CODE
				   LEFT JOIN
				   (  SELECT INVDETS.PORTFOLIO_CODE                           AS PORTFOLIO_CODE,
							 NVL (SUM (IS_TRANSACTION_DETAILS.AMOUNT), 0)     AS PAMOUNT
						FROM IS_TRANSACTION_DETAILS, IS_INVESTOR_DETAILS INVDETS
					   WHERE     IS_TRANSACTION_DETAILS.PORTFOLIO_CODE =
								 INVDETS.PORTFOLIO_CODE
							 AND IS_TRANSACTION_DETAILS.BUSINESS_DATE <=
								 :request_date
							 AND IS_TRANSACTION_DETAILS.TRANSACTION_TYPE = 'withdraw'
							 AND IS_TRANSACTION_DETAILS.HONOR_DISHONOR_CANCEL = 'p'
					GROUP BY INVDETS.PORTFOLIO_CODE) PENDING_WITHDRAW
					   ON PENDING_WITHDRAW.PORTFOLIO_CODE = INVDET.PORTFOLIO_CODE
				   
				   LEFT JOIN
				   (SELECT CMSTOCK.PORTFOLIO_CODE                AS PORTFOLIO_CODE,
							 SUM (NVL (CMSTOCK.NET_AMOUNT, 0))     AS TOTAL_SHARE_IN
						FROM IS_CAP_MKT_STOCK CMSTOCK
					   WHERE     CMSTOCK.STATUS IN ('li')
							 AND CMSTOCK.STOCK_MATURE_DATE IS NULL
							 AND CMSTOCK.BUSINESS_DATE <=  :request_date
					GROUP BY CMSTOCK.PORTFOLIO_CODE) SHARE_IN_DETAILS
					   ON SHARE_IN_DETAILS.PORTFOLIO_CODE = INVDET.PORTFOLIO_CODE
				   
				   LEFT JOIN
				   (SELECT CMSTOCK.PORTFOLIO_CODE                AS PORTFOLIO_CODE,
							 SUM (NVL (CMSTOCK.NET_AMOUNT, 0))     AS TOTAL_SHARE_OUT
						FROM IS_CAP_MKT_STOCK CMSTOCK
					   WHERE     CMSTOCK.STATUS IN ('lo')
							 AND CMSTOCK.STOCK_MATURE_DATE IS NULL
							 AND CMSTOCK.BUSINESS_DATE <=  :request_date
					GROUP BY CMSTOCK.PORTFOLIO_CODE) SHARE_OUT_DETAILS
					   ON SHARE_OUT_DETAILS.PORTFOLIO_CODE = INVDET.PORTFOLIO_CODE
					   	   
				   LEFT JOIN
				   (SELECT IDETAIL.PORTFOLIO_CODE     PORTFOLIO_CODE,
						   IC.CHARGE_VALUE            INTEREST_RATE
					  FROM IS_INVESTOR_DETAILS  IDETAIL,
						   IS_INVESTOR_CHARGE   IC,
						   LS_TEMPLATE_CHARGE   TC,
						   LS_PORTFOLIO_CHARGE  PC,
						   GS_CHARGE_LIST       CL
					 WHERE     UPPER (IDETAIL.PORTFOLIO_CODE) = UPPER (IC.PORTFOLIO_CODE)
						   AND IC.END_DATE IS NULL
						   AND IC.TEMPLATE_CHARGE_ID = TC.TEMPLATE_CHARGE_ID
						   AND TC.PORTFOLIO_CHARGE_ID = PC.PORTFOLIO_CHARGE_ID
						   AND PC.CHARGE_LIST_ID = CL.CHARGE_LIST_ID
						   AND CL.CHARGE_CODE = 4
						   AND IDETAIL.IS_TEMPLATE = 'n'
						   AND IDETAIL.ACCOUNT_CLOSE = 'n') INTEREST_DETAILS
					   ON INTEREST_DETAILS.PORTFOLIO_CODE = INVDET.PORTFOLIO_CODE
			 WHERE     INVDET.PORTFOLIO_CODE = :portfolio_code
				   AND P_PORTFOLIO_STATEMENT_TEMP.PORTFOLIO_CODE = INVDET.PORTFOLIO_CODE
				   AND INVDET.PORTFOLIO_CODE = :portfolio_code
				   AND P_PORTFOLIO_STATEMENT_TEMP.BUSINESS_DATE =
					   (SELECT MAX (P_PORTFOLIO_STATEMENT_TEMP.BUSINESS_DATE)
						  FROM P_PORTFOLIO_STATEMENT_TEMP P_PORTFOLIO_STATEMENT_TEMP
						 WHERE     P_PORTFOLIO_STATEMENT_TEMP.PORTFOLIO_CODE =
								   INVDET.PORTFOLIO_CODE
							   AND INVDET.PORTFOLIO_CODE = :portfolio_code
							   AND P_PORTFOLIO_STATEMENT_TEMP.BUSINESS_DATE <=
								   :request_date)
		";

		$sql = DB::select(
			$sql,
			[
				'request_date' => $requestDate->format('Y-m-d'),
				'portfolio_code' =>$portfolioCode
			]
		);

		return $sql;
	}

	private function getCashDivident($portfolioCode, $requestDate)
	{
		$dividendSql = "
			SELECT COALESCE (INVDET.INSTITUTION_NAME, INVDET.INVESTOR_NAME)
					   AS INVESTOR_NAME,
				   INVDET.PORTFOLIO_CODE,
				   INVDET.BO_ACCOUNT_NO,
				   INVDET.MOBILE,
				   NVL (INTEREST_DETAILS.INTEREST_RATE, 0)
					   AS INTEREST_RATE,
				   INVDET.EMAIL,
				   INVDET.TEMPLATE_ID,
				   INVDET.TENURE_YEAR,
				   INVDET.TENURE_MATURE_DATE,
				   INVDET.MONTHLY_SCHEME,
				   INVDET.BUSINESS_DATE
					   AS DATE_OF_ACC_OPENING,
				   INVDET.ACCOUNT_CLOSE,
				   INVDET.ACCOUNT_SUSPEND,
				   INVDET.ACCOUNT_ENABLE_DISABLE,
				   (CASE
						WHEN INVDET.IS_MARGIN_LOAN_ACTIVE = 'y' THEN 'Margin'
						ELSE 'Non Margin'
					END)
					   AS MARGIN_NONMARGIN,
				   NVL ((P_PORTFOLIO_STATEMENT_TEMP.PURCHASE_POWER), 0)
					   AS PURCHASE_POWER,
				   ''
					   AS PURCHASE_POWER_DR_CR,
				   NVL ((P_PORTFOLIO_STATEMENT_TEMP.RECEIVABLES_SALES), 0)
					   AS RECIEVABLE_SALES,
				   ''
					   AS RECIEVABLE_SALES_DR_CR,
				   NVL ((P_PORTFOLIO_STATEMENT_TEMP.TOTAL_DEPOSIT), 0)
					   AS TOTAL_DEPOSIT,
				   NVL ((REALIZED_GAINLOSS.REALIZED_GAIN_LOSS), 0)
					   AS REALIZED_GAINLOSS,
				   ''
					   AS REALIZED_GAINLOSS_DR_CR,
				   0
					   AS ROI,
				   ''
					   AS ROI_DR_CR,
				   NVL ((P_PORTFOLIO_STATEMENT_TEMP.DIVIDEND_INCOME), 0)
					   AS DIVIDEND_INCOME,
				   NVL ((P_PORTFOLIO_STATEMENT_TEMP.WITHDRAW), 0)
					   AS WITHDRAW,
				   NVL ((P_PORTFOLIO_STATEMENT_TEMP.MARKET_VALUE_SECURITIES), 0)
					   AS MKT_VALUE_SEC,
				   NVL ((P_PORTFOLIO_STATEMENT_TEMP.UNREALIZE_GAIN_LOSS), 0)
					   AS UNREALIZE_GL,
				   ''
					   AS UNREALIZE_GL_DR_CR,
				   NVL ((P_PORTFOLIO_STATEMENT_TEMP.EQUITY), 0)
					   AS EQUITY,
				   ''
					   AS EQUITY_DR_CR,
				   NVL ((P_PORTFOLIO_STATEMENT_TEMP.FUND_TRANSFER_IN), 0)
					   AS TRANSFERIN,
				   NVL ((P_PORTFOLIO_STATEMENT_TEMP.FUND_TRANSFER_OUT), 0)
					   AS TRANSFEROUT,
				   NVL ((P_PORTFOLIO_STATEMENT_TEMP.ACCRUED_FEES_CHARGE), 0)
					   AS ACCRUED,
				   NVL ((P_PORTFOLIO_STATEMENT_TEMP.EQUITY_TO_DEBT_RATIO), 0)
					   AS EQUITY_TO_DEBT_RATIO,
				   NVL ((P_PORTFOLIO_STATEMENT_TEMP.RECEIVABLE_DIVIDEND), 0)
					   AS CASH_DIV_RECEIVABLE,
				   NVL ((P_PORTFOLIO_STATEMENT_TEMP.RECEIVED_DIVIDEND), 0)
					   AS CASH_DIV_RECEIVED,
				   NVL ((P_PORTFOLIO_STATEMENT_TEMP.LOAN_RATIO), 0)
					   AS LOAN_RATIO,
				   NVL ((P_PORTFOLIO_STATEMENT_TEMP.ACCRUED_FEES_CHARGE), 0)
					   AS ACCRUED_FEES_CHARGE,
				   NVL ((P_PORTFOLIO_STATEMENT_TEMP.CURRENT_BALANCE), 0)
					   AS CURRENT_BALANCE,
				   SHARE_IN_DETAILS.TOTAL_SHARE_IN
					   AS SHARE_TRANSFER_IN,
				   SHARE_OUT_DETAILS.TOTAL_SHARE_OUT
					   AS SHARE_TRANSFER_OUT,
				   NVL ((PENDING_WITHDRAW.PAMOUNT), 0)
					   AS PAMOUNT,
				   NVL ((P_PORTFOLIO_STATEMENT_TEMP.IPO_ORDER_AMOUNT), 0)
					   AS IPO_ORDER_AMOUNT
			  FROM P_PORTFOLIO_STATEMENT_TEMP  P_PORTFOLIO_STATEMENT_TEMP,
				   IS_INVESTOR_DETAILS         INVDET
				   LEFT JOIN
				   (  SELECT X.PORTFOLIO_CODE, SUM (x.BROKER_COMMISSION) REALIZED_GAIN_LOSS
						FROM (  SELECT CM_STOCK.PORTFOLIO_CODE,
									   SUM (
										   (  (CM_STOCK.SHARES * CM_STOCK.ACTUAL_RATE)
											- (CM_STOCK.SHARES * CM_STOCK.AVG_RATE)
											- (CM_STOCK.BROKER_COMMISSION)))
										   AS BROKER_COMMISSION
								  FROM IS_CAP_MKT_STOCK CM_STOCK
								 WHERE     CM_STOCK.STATUS = 'enc'
									   AND CM_STOCK.STOCK_MATURE_DATE IS NULL
									   AND CM_STOCK.BUSINESS_DATE <=
										   :business_date
							  GROUP BY CM_STOCK.PORTFOLIO_CODE, CM_STOCK.BUSINESS_DATE) X
					GROUP BY X.PORTFOLIO_CODE) REALIZED_GAINLOSS
					   ON REALIZED_GAINLOSS.PORTFOLIO_CODE = INVDET.PORTFOLIO_CODE
				   LEFT JOIN
				   (  SELECT INVDETS.PORTFOLIO_CODE                           AS PORTFOLIO_CODE,
							 NVL (SUM (IS_TRANSACTION_DETAILS.AMOUNT), 0)     AS PAMOUNT
						FROM IS_TRANSACTION_DETAILS, IS_INVESTOR_DETAILS INVDETS
					   WHERE     IS_TRANSACTION_DETAILS.PORTFOLIO_CODE =
								 INVDETS.PORTFOLIO_CODE
							 AND IS_TRANSACTION_DETAILS.BUSINESS_DATE <=
								 :business_date
							 AND IS_TRANSACTION_DETAILS.TRANSACTION_TYPE = 'withdraw'
							 AND IS_TRANSACTION_DETAILS.HONOR_DISHONOR_CANCEL = 'p'
					GROUP BY INVDETS.PORTFOLIO_CODE) PENDING_WITHDRAW
					   ON PENDING_WITHDRAW.PORTFOLIO_CODE = INVDET.PORTFOLIO_CODE
				   
				   LEFT JOIN
				   (SELECT CMSTOCK.PORTFOLIO_CODE                AS PORTFOLIO_CODE,
							 SUM (NVL (CMSTOCK.NET_AMOUNT, 0))     AS TOTAL_SHARE_IN
						FROM IS_CAP_MKT_STOCK CMSTOCK
					   WHERE     CMSTOCK.STATUS IN ('li')
							 AND CMSTOCK.STOCK_MATURE_DATE IS NULL
							 AND CMSTOCK.BUSINESS_DATE <=  :business_date
					GROUP BY CMSTOCK.PORTFOLIO_CODE) SHARE_IN_DETAILS
					   ON SHARE_IN_DETAILS.PORTFOLIO_CODE = INVDET.PORTFOLIO_CODE
				   
				   LEFT JOIN
				   (SELECT CMSTOCK.PORTFOLIO_CODE                AS PORTFOLIO_CODE,
							 SUM (NVL (CMSTOCK.NET_AMOUNT, 0))     AS TOTAL_SHARE_OUT
						FROM IS_CAP_MKT_STOCK CMSTOCK
					   WHERE     CMSTOCK.STATUS IN ('lo')
							 AND CMSTOCK.STOCK_MATURE_DATE IS NULL
							 AND CMSTOCK.BUSINESS_DATE <=  :business_date
					GROUP BY CMSTOCK.PORTFOLIO_CODE) SHARE_OUT_DETAILS
					   ON SHARE_OUT_DETAILS.PORTFOLIO_CODE = INVDET.PORTFOLIO_CODE
					   	   
				   LEFT JOIN
				   (SELECT IDETAIL.PORTFOLIO_CODE     PORTFOLIO_CODE,
						   IC.CHARGE_VALUE            INTEREST_RATE
					  FROM IS_INVESTOR_DETAILS  IDETAIL,
						   IS_INVESTOR_CHARGE   IC,
						   LS_TEMPLATE_CHARGE   TC,
						   LS_PORTFOLIO_CHARGE  PC,
						   GS_CHARGE_LIST       CL
					 WHERE     UPPER (IDETAIL.PORTFOLIO_CODE) = UPPER (IC.PORTFOLIO_CODE)
						   AND IC.END_DATE IS NULL
						   AND IC.TEMPLATE_CHARGE_ID = TC.TEMPLATE_CHARGE_ID
						   AND TC.PORTFOLIO_CHARGE_ID = PC.PORTFOLIO_CHARGE_ID
						   AND PC.CHARGE_LIST_ID = CL.CHARGE_LIST_ID
						   AND CL.CHARGE_CODE = 4
						   AND IDETAIL.IS_TEMPLATE = 'n'
						   AND IDETAIL.ACCOUNT_CLOSE = 'n') INTEREST_DETAILS
					   ON INTEREST_DETAILS.PORTFOLIO_CODE = INVDET.PORTFOLIO_CODE
			 WHERE     INVDET.PORTFOLIO_CODE = :portfolio_code
				   AND P_PORTFOLIO_STATEMENT_TEMP.PORTFOLIO_CODE = INVDET.PORTFOLIO_CODE
				   AND INVDET.PORTFOLIO_CODE = :portfolio_code
				   AND P_PORTFOLIO_STATEMENT_TEMP.BUSINESS_DATE =
					   (SELECT MAX (P_PORTFOLIO_STATEMENT_TEMP.BUSINESS_DATE)
						  FROM P_PORTFOLIO_STATEMENT_TEMP P_PORTFOLIO_STATEMENT_TEMP
						 WHERE     P_PORTFOLIO_STATEMENT_TEMP.PORTFOLIO_CODE =
								   INVDET.PORTFOLIO_CODE
							   AND INVDET.PORTFOLIO_CODE = :portfolio_code
							   AND P_PORTFOLIO_STATEMENT_TEMP.BUSINESS_DATE <=
								   :business_date)
		";

		$dividendResult = DB::select(
			$dividendSql,
			[
				'business_date'  => $requestDate->format('Y-m-d'),
				'portfolio_code' => $portfolioCode,
			]
		);

		return $dividendResult;
	}

	private function getStockDivident($portfolioCode, $requestDate)
	{
		$dividendSql = "
			SELECT COALESCE (INVDET.INSTITUTION_NAME, INVDET.INVESTOR_NAME)
					   AS INVESTOR_NAME,
				   INVDET.PORTFOLIO_CODE,
				   INVDET.BO_ACCOUNT_NO,
				   INVDET.MOBILE,
				   NVL (INTEREST_DETAILS.INTEREST_RATE, 0)
					   AS INTEREST_RATE,
				   INVDET.EMAIL,
				   INVDET.TEMPLATE_ID,
				   INVDET.TENURE_YEAR,
				   INVDET.TENURE_MATURE_DATE,
				   INVDET.MONTHLY_SCHEME,
				   INVDET.BUSINESS_DATE
					   AS DATE_OF_ACC_OPENING,
				   INVDET.ACCOUNT_CLOSE,
				   INVDET.ACCOUNT_SUSPEND,
				   INVDET.ACCOUNT_ENABLE_DISABLE,
				   (CASE
						WHEN INVDET.IS_MARGIN_LOAN_ACTIVE = 'y' THEN 'Margin'
						ELSE 'Non Margin'
					END)
					   AS MARGIN_NONMARGIN,
				   NVL ((P_PORTFOLIO_STATEMENT_TEMP.PURCHASE_POWER), 0)
					   AS PURCHASE_POWER,
				   ''
					   AS PURCHASE_POWER_DR_CR,
				   NVL ((P_PORTFOLIO_STATEMENT_TEMP.RECEIVABLES_SALES), 0)
					   AS RECIEVABLE_SALES,
				   ''
					   AS RECIEVABLE_SALES_DR_CR,
				   NVL ((P_PORTFOLIO_STATEMENT_TEMP.TOTAL_DEPOSIT), 0)
					   AS TOTAL_DEPOSIT,
				   NVL ((REALIZED_GAINLOSS.REALIZED_GAIN_LOSS), 0)
					   AS REALIZED_GAINLOSS,
				   ''
					   AS REALIZED_GAINLOSS_DR_CR,
				   0
					   AS ROI,
				   ''
					   AS ROI_DR_CR,
				   NVL ((P_PORTFOLIO_STATEMENT_TEMP.DIVIDEND_INCOME), 0)
					   AS DIVIDEND_INCOME,
				   NVL ((P_PORTFOLIO_STATEMENT_TEMP.WITHDRAW), 0)
					   AS WITHDRAW,
				   NVL ((P_PORTFOLIO_STATEMENT_TEMP.MARKET_VALUE_SECURITIES), 0)
					   AS MKT_VALUE_SEC,
				   NVL ((P_PORTFOLIO_STATEMENT_TEMP.UNREALIZE_GAIN_LOSS), 0)
					   AS UNREALIZE_GL,
				   ''
					   AS UNREALIZE_GL_DR_CR,
				   NVL ((P_PORTFOLIO_STATEMENT_TEMP.EQUITY), 0)
					   AS EQUITY,
				   ''
					   AS EQUITY_DR_CR,
				   NVL ((P_PORTFOLIO_STATEMENT_TEMP.FUND_TRANSFER_IN), 0)
					   AS TRANSFERIN,
				   NVL ((P_PORTFOLIO_STATEMENT_TEMP.FUND_TRANSFER_OUT), 0)
					   AS TRANSFEROUT,
				   NVL ((P_PORTFOLIO_STATEMENT_TEMP.ACCRUED_FEES_CHARGE), 0)
					   AS ACCRUED,
				   NVL ((P_PORTFOLIO_STATEMENT_TEMP.EQUITY_TO_DEBT_RATIO), 0)
					   AS EQUITY_TO_DEBT_RATIO,
				   NVL ((P_PORTFOLIO_STATEMENT_TEMP.RECEIVABLE_DIVIDEND), 0)
					   AS CASH_DIV_RECEIVABLE,
				   NVL ((P_PORTFOLIO_STATEMENT_TEMP.RECEIVED_DIVIDEND), 0)
					   AS CASH_DIV_RECEIVED,
				   NVL ((P_PORTFOLIO_STATEMENT_TEMP.LOAN_RATIO), 0)
					   AS LOAN_RATIO,
				   NVL ((P_PORTFOLIO_STATEMENT_TEMP.ACCRUED_FEES_CHARGE), 0)
					   AS ACCRUED_FEES_CHARGE,
				   NVL ((P_PORTFOLIO_STATEMENT_TEMP.CURRENT_BALANCE), 0)
					   AS CURRENT_BALANCE,
				   SHARE_IN_DETAILS.TOTAL_SHARE_IN
					   AS SHARE_TRANSFER_IN,
				   SHARE_OUT_DETAILS.TOTAL_SHARE_OUT
					   AS SHARE_TRANSFER_OUT,
				   NVL ((PENDING_WITHDRAW.PAMOUNT), 0)
					   AS PAMOUNT,
				   NVL ((P_PORTFOLIO_STATEMENT_TEMP.IPO_ORDER_AMOUNT), 0)
					   AS IPO_ORDER_AMOUNT
			  FROM P_PORTFOLIO_STATEMENT_TEMP  P_PORTFOLIO_STATEMENT_TEMP,
				   IS_INVESTOR_DETAILS         INVDET
				   LEFT JOIN
				   (  SELECT X.PORTFOLIO_CODE, SUM (x.BROKER_COMMISSION) REALIZED_GAIN_LOSS
						FROM (  SELECT CM_STOCK.PORTFOLIO_CODE,
									   SUM (
										   (  (CM_STOCK.SHARES * CM_STOCK.ACTUAL_RATE)
											- (CM_STOCK.SHARES * CM_STOCK.AVG_RATE)
											- (CM_STOCK.BROKER_COMMISSION)))
										   AS BROKER_COMMISSION
								  FROM IS_CAP_MKT_STOCK CM_STOCK
								 WHERE     CM_STOCK.STATUS = 'enc'
									   AND CM_STOCK.STOCK_MATURE_DATE IS NULL
									   AND CM_STOCK.BUSINESS_DATE <=
										   :business_date
							  GROUP BY CM_STOCK.PORTFOLIO_CODE, CM_STOCK.BUSINESS_DATE) X
					GROUP BY X.PORTFOLIO_CODE) REALIZED_GAINLOSS
					   ON REALIZED_GAINLOSS.PORTFOLIO_CODE = INVDET.PORTFOLIO_CODE
				   LEFT JOIN
				   (  SELECT INVDETS.PORTFOLIO_CODE                           AS PORTFOLIO_CODE,
							 NVL (SUM (IS_TRANSACTION_DETAILS.AMOUNT), 0)     AS PAMOUNT
						FROM IS_TRANSACTION_DETAILS, IS_INVESTOR_DETAILS INVDETS
					   WHERE     IS_TRANSACTION_DETAILS.PORTFOLIO_CODE =
								 INVDETS.PORTFOLIO_CODE
							 AND IS_TRANSACTION_DETAILS.BUSINESS_DATE <=
								 :business_date
							 AND IS_TRANSACTION_DETAILS.TRANSACTION_TYPE = 'withdraw'
							 AND IS_TRANSACTION_DETAILS.HONOR_DISHONOR_CANCEL = 'p'
					GROUP BY INVDETS.PORTFOLIO_CODE) PENDING_WITHDRAW
					   ON PENDING_WITHDRAW.PORTFOLIO_CODE = INVDET.PORTFOLIO_CODE
				   
				   LEFT JOIN
				   (SELECT CMSTOCK.PORTFOLIO_CODE                AS PORTFOLIO_CODE,
							 SUM (NVL (CMSTOCK.NET_AMOUNT, 0))     AS TOTAL_SHARE_IN
						FROM IS_CAP_MKT_STOCK CMSTOCK
					   WHERE     CMSTOCK.STATUS IN ('li')
							 AND CMSTOCK.STOCK_MATURE_DATE IS NULL
							 AND CMSTOCK.BUSINESS_DATE <=  :business_date
					GROUP BY CMSTOCK.PORTFOLIO_CODE) SHARE_IN_DETAILS
					   ON SHARE_IN_DETAILS.PORTFOLIO_CODE = INVDET.PORTFOLIO_CODE
				   
				   LEFT JOIN
				   (SELECT CMSTOCK.PORTFOLIO_CODE                AS PORTFOLIO_CODE,
							 SUM (NVL (CMSTOCK.NET_AMOUNT, 0))     AS TOTAL_SHARE_OUT
						FROM IS_CAP_MKT_STOCK CMSTOCK
					   WHERE     CMSTOCK.STATUS IN ('lo')
							 AND CMSTOCK.STOCK_MATURE_DATE IS NULL
							 AND CMSTOCK.BUSINESS_DATE <=  :business_date
					GROUP BY CMSTOCK.PORTFOLIO_CODE) SHARE_OUT_DETAILS
					   ON SHARE_OUT_DETAILS.PORTFOLIO_CODE = INVDET.PORTFOLIO_CODE
					   	   
				   LEFT JOIN
				   (SELECT IDETAIL.PORTFOLIO_CODE     PORTFOLIO_CODE,
						   IC.CHARGE_VALUE            INTEREST_RATE
					  FROM IS_INVESTOR_DETAILS  IDETAIL,
						   IS_INVESTOR_CHARGE   IC,
						   LS_TEMPLATE_CHARGE   TC,
						   LS_PORTFOLIO_CHARGE  PC,
						   GS_CHARGE_LIST       CL
					 WHERE     UPPER (IDETAIL.PORTFOLIO_CODE) = UPPER (IC.PORTFOLIO_CODE)
						   AND IC.END_DATE IS NULL
						   AND IC.TEMPLATE_CHARGE_ID = TC.TEMPLATE_CHARGE_ID
						   AND TC.PORTFOLIO_CHARGE_ID = PC.PORTFOLIO_CHARGE_ID
						   AND PC.CHARGE_LIST_ID = CL.CHARGE_LIST_ID
						   AND CL.CHARGE_CODE = 4
						   AND IDETAIL.IS_TEMPLATE = 'n'
						   AND IDETAIL.ACCOUNT_CLOSE = 'n') INTEREST_DETAILS
					   ON INTEREST_DETAILS.PORTFOLIO_CODE = INVDET.PORTFOLIO_CODE
			 WHERE     INVDET.PORTFOLIO_CODE = :portfolio_code
				   AND P_PORTFOLIO_STATEMENT_TEMP.PORTFOLIO_CODE = INVDET.PORTFOLIO_CODE
				   AND INVDET.PORTFOLIO_CODE = :portfolio_code
				   AND P_PORTFOLIO_STATEMENT_TEMP.BUSINESS_DATE =
					   (SELECT MAX (P_PORTFOLIO_STATEMENT_TEMP.BUSINESS_DATE)
						  FROM P_PORTFOLIO_STATEMENT_TEMP P_PORTFOLIO_STATEMENT_TEMP
						 WHERE     P_PORTFOLIO_STATEMENT_TEMP.PORTFOLIO_CODE =
								   INVDET.PORTFOLIO_CODE
							   AND INVDET.PORTFOLIO_CODE = :portfolio_code
							   AND P_PORTFOLIO_STATEMENT_TEMP.BUSINESS_DATE <=
								   :business_date)
		";
		$dividendResult = DB::select(
			$dividendSql,
			[
				'business_date'  => $requestDate->format('Y-m-d'),
				'portfolio_code' => $portfolioCode,
			]
		);

		return $dividendResult;
	}

	private function getIpoInformation($portfolioCode, $requestDate)
	{
		$ipoSql = "
			SELECT 
				DISTINCT(IPOD.INSTRUMENT_NAME) AS INSTRUMENT_NAME,
				IPOD.INSTRUMENT_LOT_SIZE,
				IPOA.RATE,
				IPOAA.ALLOCATED_SHARE,
				IPOA.CURRENT_BALANCE,
				IPOA.PURCHASE_POWER,
				IPOA.TOTAL_AMOUNT,
				IPOD.ACTUAL_DEC_DATE,
				IPOD.ACTUAL_LA_DATE,
				IPOD.BUSINESS_DATE
			FROM IS_IPO_DETAILS IPOD, IS_IPO_ORDER IPOO, IS_IPO_APPLICATION IPOA, IS_IPO_ALLOCATION IPOAA
			WHERE IPOD.IPO_DETAILS_ID = IPOO.IPO_DETAILS_ID
			AND IPOO.IPO_ORDER_ID = IPOA.IPO_ORDER_ID
			AND IPOA.IPO_APPLICATION_ID = IPOAA.IPO_APPLICATION_ID
			AND IPOAA.PORTFOLIO_CODE = :portfolio_code
			AND IPOAA.BUSINESS_DATE <= :business_date
			AND lower(IPOAA.STATUS) = 'n'
		";
		$ipoSqlResult = DB::select(
			$ipoSql,
			[
				'business_date'  => $requestDate->format('Y-m-d'),
				'portfolio_code' => $portfolioCode,
			]
		);

		return $ipoSqlResult;
	}

	private function getUnClearChequeTotAmount($portfolioCode, $requestDate)
	{
		$sql = "
			SELECT NVL(SUM(TRANDET.AMOUNT), 0) AS TOTALUNCLEARCHQ
			FROM 
				IS_TRANSACTION_DETAILS TRANDET
			WHERE 
				TRANDET.PORTFOLIO_CODE = :portfolio_code
			AND TRANDET.HONOR_DISHONOR_CANCEL = 'p'
			AND TRANDET.BUSINESS_DATE <= :business_date
							   
		";
		$sqlResult = DB::select(
			$sql,
			[
				'business_date'  => $requestDate->format('Y-m-d'),
				'portfolio_code' => $portfolioCode,
			]
		);

		return $sqlResult;
	}

	private function getRightShareApplliedTotAmount($portfolioCode, $requestDate)
	{
		$sql = "
			SELECT 
				NVL(SUM(RSAPP.AMOUNT), 0) AS TOTALRSAPPLY
			FROM 
				IS_RIGHT_SHARE_APPLICATION RSAPP
			WHERE 
				RSAPP.PORTFOLIO_CODE = :portfolio_code
			AND RSAPP.STATUS = 'n'
			AND RSAPP.BUSINESS_DATE <= :business_date
							   
		";
		$sqlResult = DB::select(
			$sql,
			[
				'business_date'  => $requestDate->format('Y-m-d'),
				'portfolio_code' => $portfolioCode,
			]
		);

		return $sqlResult;
	}

	private function getIPOApplliedTotAmount($portfolioCode, $requestDate)
	{
		$sql = "
			SELECT 
				SUM(PORTTB.CB_AMOUNT) AS TOTALIPOAPPLY
			FROM 
				IS_PORTFOLIO_TRIAL_BAL PORTTB
			WHERE 
				PORTTB.AC_CODE LIKE '302001%'
			AND PORTTB.PORTFOLIO_CODE = :portfolio_code
			AND PORTTB.BALANCE_DATE = :business_date
							   
		";
		$sqlResult = DB::select(
			$sql,
			[
				'business_date'  => $requestDate->format('Y-m-d'),
				'portfolio_code' => $portfolioCode,
			]
		);

		return $sqlResult;
	}

	
}
