'use client'
import React, { useState, useEffect, useRef } from 'react';
import styles from "../../assets/style.module.css";
import { useSession } from "next-auth/react";
import Loading from '../loading';
import Swal from 'sweetalert2'
import { signOut } from 'next-auth/react';

const ClientComponent = () => {
    const [investorPoftfolio, setInvestorPortfolio] = useState([])
    const [isLoading, setIsLoading] = useState(false)
    const { data } = useSession();
    const prevToken = useRef(null);

    useEffect(() => {
        let token;
        if (data && data.user && data.user.token && data.user.token !== prevToken.current) {
            token = data.user.token;
            prevToken.current = token
            setIsLoading(true);
            fetchPortfolioData(token);
        }

    }, [data?.user?.token]);

    function formatIndianNumber(value) {
        if (value === null || value === undefined || isNaN(value)) {
            return "0";
        }
    
        const isNegative = value < 0; // Check if the number is negative
        const absoluteValue = Math.abs(value).toString(); // Get the absolute value
    
        const parts = absoluteValue.split("."); // Split into whole and fractional parts
        const wholePart = parts[0];
        const fractionalPart = parts[1] ? `.${parts[1]}` : "";
    
        // Correct regex for Indian numbering system
        const formattedWholePart = wholePart.replace(
            /(\d+)(\d{3})$/,
            (match, p1, p2) => p1.replace(/\B(?=(\d{2})+(?!\d))/g, ",") + "," + p2
        );
    
        const formattedNumber = formattedWholePart + fractionalPart;
    
        return isNegative ? `(${formattedNumber})` : formattedNumber; // Add parentheses for negative numbers
    }
    
    

    const fetchPortfolioData = async (token) => {
        try {
            const url = process.env.NEXT_PUBLIC_API_URL + '/dashboard-summary-info';
            const res = await fetch(url, {
                method: "GET",
                headers: {
                    "Accept": "application/json",
                    'Content-Type': 'application/json',
                    'Authorization': 'Bearer ' + String(token)
                }
            });
            const data = await res.json();
            console.log(data);
            if (data.status == 'success') {
                console.log(data.dashboard_summary);
                setInvestorPortfolio(data.dashboard_summary[0])
            } else if (res.status == 401) {
                Swal.fire({
                    position: "top-end",
                    icon: "warning",
                    title: data.message,
                    showConfirmButton: false,
                    timer: 1000,
                }).then((result) => {
                    if (result.dismiss === Swal.DismissReason.timer) {
                        signOut();
                    }
                });
            }
        } catch (error) {
            if (error.name == 'TypeError') {
                Swal.fire({
                    position: "top-end",
                    icon: "error",
                    title: 'The server is not responding. Please wait for a moment.',
                    showConfirmButton: false,
                    timer: 2000,
                });
            }
        } finally {
            setIsLoading(false)
        }
    };

    if (isLoading) {
        return (
            <Loading loading={isLoading} />
        )
    }

    return (
        <>
                {
                    investorPoftfolio && Object.keys(investorPoftfolio).length > 0 && (
                    <React.Fragment>
                        <div className={`${styles['welcome-section']}`}>
                            <div className='row'>
                                <div className='col-12'>
                                    <p className='text-end'>
                                        {
                                            ` ${new Date().getDate()}-${new Date().toLocaleString('default', { month: 'long' })}-${new Date().getFullYear()} `
                                        }
                                    </p>
                                </div>
                            </div>
                            <h2>Welcome<br/> Mr/Ms. {investorPoftfolio.investor_name}!</h2>
                            <p>Financial Assistance With True Purpose</p>
                        </div>


                        <div className={'row mb-4 card-wrapper'}>
                            <div className={'col-md-4 col-lg-2'}>
                                <div className={`${styles['card-box']}`}>
                                    <h5 className={styles.text_color}>Cost Value</h5>
                                    <p> {formatIndianNumber(parseFloat(investorPoftfolio.cost_value_securities || 0))}</p>
                                </div>
                            </div>

                            <div className={'col-md-4 col-lg-2'}>
                                <div className={`${styles['card-box']}`}>
                                    <h5 className={styles.text_color}>Market Value</h5>
                                    <p> {formatIndianNumber(parseFloat(investorPoftfolio.market_value_securities || 0))}</p>
                                </div>
                            </div>

                            <div className={'col-md-4 col-lg-2'}>
                                <div className={`${styles['card-box']}`}>
                                    <h5 className={styles.text_color}>Equity</h5>
                                    <p> {formatIndianNumber(parseFloat(investorPoftfolio.equity_at_market || 0))}</p>
                                </div>
                            </div>

                            <div className={'col-md-4 col-lg-2'}>
                                <div className={`${styles['card-box']}`}>
                                    <h5 className={styles.text_color}>Purchase Power</h5>
                                    <p> {formatIndianNumber(parseFloat(investorPoftfolio.purchase_power || 0))}</p>
                                </div>
                            </div>

                            <div className={'col-md-4 col-lg-2'}>
                                <div className={`${styles['card-box']}`}>
                                    <h5 className={styles.text_color}>CMR</h5>
                                    <p>{investorPoftfolio.cmr_ration + '%' || "N/A"}</p>
                                </div>
                            </div>

                            <div className={'col-md-4 col-lg-2'}>
                                <div className={`${styles['card-box']}`}>
                                    <h5 className={styles.text_color}>Net Gain/Loss</h5>
                                    <p>+ {formatIndianNumber(parseFloat(investorPoftfolio.net_gain_loss || 0))}</p>
                                </div>
                            </div>

                        </div>
                    </React.Fragment>                     
                )}
        </>
    );
};

export default ClientComponent;
