<?php

namespace App\Http\Controllers\Auth;

use Carbon\Carbon;
use App\Models\User;
use App\Models\NomineeInfo;
use App\Models\BusinessDate;
use App\Models\MessageRoute;
use Illuminate\Http\Request;
use Ramsey\Uuid\Type\Integer;
use App\Models\InvestorDetail;
use App\Models\TransactionDetails;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;

class AuthController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth:api', ['except' => ['login', 'registation', 'signUp', 'forgotPassword']]);
    }

    public function login(Request $request)
    {
        $request->validate([
            'email'    => 'required|string|email',
            'password' => 'required|string',
        ]);
        
        $credentials = $request->only('email', 'password');

        $token = Auth::attempt($credentials);

        if (!$token) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Unauthorized',
            ], 401);
        }

        $user = Auth::user();
        return response()->json([
            'status'        => 'success',
            'user'          => $user,
            'authorization' => [
                'token' => $token,
                'type'  => 'bearer',
            ]
        ]);

    }

    public function logout()
    {
        Auth::logout();
        return response()->json([
            'status'  => 'success',
            'message' => 'Successfully logged out',
        ]);
    }

    public function refresh()
    {
        return response()->json([
            'status'        => 'success',
            'user'          => Auth::user(),
            'authorisation' => [
                'token' => Auth::refresh(),
                'type'  => 'bearer',
            ]
        ]);
    }

    public function registation(Request $request, int $pageId)
    {
        try {
            $businessDate = BusinessDate::latest('business_date')->first();

            if ($pageId === 1) {
                      /**
                 * Find the investor with email or mobile
                 */
                $validator = Validator::make(
                    $request->all(),
                    [
                        'email'          => 'required|email',
                        'contact_number' => 'required|digits:11',
                    ],
                    [
                        'email.required'          => 'The email field is required',
                        'email.email'             => 'The email field is not a valid email address',
                        'contact_number.required' => 'The contact number field is required',
                        'contact_number.digit'    => 'The contact number must be 11 digit',
                              // 'email.unique'    => "The email is already registered",
                    ]
                );

                if ($validator->fails()) {
                    return response()->json([
                        'status'  => 'error',
                        'message' => 'Form validation error',
                        'errors'  => $validator->errors()->messages(),
                    ], 400);
                } else {
                    $investor_details_arr = [];
                    $nominee_details_arr  = [];
                    $investorDetail       = InvestorDetail::with(['nomineeInfo'])
                    ->where('email', $request->email)
                    ->orWhere('mobile', $request->contact_number)
                    ->get();

                    if ($investorDetail->count() > 1) {
                        return response()->json([
                            'status'  => 'error',
                            'message' => 'This email or mobile is already registered',
                        ], 400);
                    }

                    $investorDetail = $investorDetail->first();

                    if (!is_null($investorDetail)) {
                        $nomineeInfo = $investorDetail->nomineeInfo;
                              // dd($nomineeInfo);

                        $investor_details_arr = [
                            'investor_details_id'   => $investorDetail->investor_details_id,
                            'investor_name'         => $investorDetail->investor_name,
                            'product_id'            => $investorDetail->template_id,
                            'date_of_birth'         => is_null($investorDetail->dob) ? null : Carbon::createFromFormat('Y-m-d H:i:s', $investorDetail->dob)->format('d-m-Y'),
                            'gender'                => $investorDetail->gender,
                            'father_name'           => $investorDetail->father_name,
                            'mother_name'           => $investorDetail->mother_name,
                            'email'                 => $investorDetail->email,
                            'contact_number'        => $investorDetail->mobile,
                            'residency'             => $investorDetail->residency,
                            'nid_number'            => $investorDetail->nid,
                            'occupation_id'         => $investorDetail->occupation_id,
                            'nationality'           => $investorDetail->nationality_id,
                            'present_address'       => $investorDetail->mailing_address,
                            'permanent_address'     => $investorDetail->permanent_address,
                            'investor_photo'        => $investorDetail->applicant_photo,
                            'investor_nid_photo'    => $investorDetail->nid_copy,
                            'investor_signature'    => $investorDetail->applicant_sign,
                            'investor_cheque_photo' => $investorDetail->bank_copy,
                            'reg_page_no'           => $investorDetail->reg_page_no,
                            'reg_is_complete'       => $investorDetail->reg_is_complete,
                        ];

                        if (!is_null($nomineeInfo)) {
                            foreach ($nomineeInfo as $index => $singleNomineeInfo) {
                                $nominee_details_arr[] = [                        
                                    'nominee_info_id'         => $singleNomineeInfo->nominee_info_id,
                                    'nominee_full_name'       => $singleNomineeInfo->nominee_name,
                                    'nominee_share'           => $singleNomineeInfo->share_percentage,
                                    'nominee_date_of_birth'   => is_null($singleNomineeInfo->dob) ? null : Carbon::createFromFormat('Y-m-d H:i:s', $singleNomineeInfo->dob)->format('d-m-Y'),
                                    'nominee_father_name'     => $singleNomineeInfo->father_name,
                                    'nominee_mother_name'     => $singleNomineeInfo->mother_name,
                                    'nominee_spouse_name'     => $singleNomineeInfo->spouse_name,
                                    'nominee_contact_number'  => $singleNomineeInfo->nominee_mobile,
                                    'nominee_nid'             => $singleNomineeInfo->nid,
                                    'nominee_mailing_address' => $singleNomineeInfo->mailing_address,
                                    'nominee_occupation_id'   => $singleNomineeInfo->occupation_id,
                                    'nominee_relationship_id' => $singleNomineeInfo->relation_id,
                                    'nominee_photo'           => $singleNomineeInfo->nominee_applicant_photo,
                                    'nominee_nid_photo'       => $singleNomineeInfo->nominee_nid,
                                    'nominee_signature'       => $singleNomineeInfo->nominee_applicant_sign,
                                ];
                            }
                        }
                    }

                    return response()->json([
                        'status'          => 'success',
                        'investor_detail' => $investor_details_arr,
                        'nominee_detail'  => $nominee_details_arr,
                    ]);
                }
            } elseif ($pageId === 2) {
                      /**
                 * verify otp
                 */
                $validator = Validator::make(
                    $request->all(),
                    [
                        'otp_code' => 'required|numeric|digits:6',
                    ],
                    [
                        'otp_code.required' => 'The otp field is required',
                        'otp_code.numeric'  => 'The otp field must be a number',
                        'otp_code.digits'   => 'The otp field must be 6 digits',
                    ]
                );
        
                if ($validator->fails()) {
                    return response()->json([
                        'status'  => 'error',
                        'message' => 'Form validation error',
                        'errors'  => $validator->errors()->messages(),
                    ], 400);
                }
        
                if (true) {
                    return response()->json([
                        'status'     => 'success',
                        'otp_verify' => true,
                    ]);
                }
        
                return response()->json([
                    'status'     => 'error',
                    'otp_verify' => false,
                ]);
            } elseif ($pageId === 3) {
                      /**
                 * Insert/update investor details
                 */
                $validator = Validator::make(
                    $request->all(),
                    [
                        'investor_details_id' => 'nullable',
                        'investor_name'       => 'required',
                        'product_id'          => "required|exists:ls_template,template_id",
                        'date_of_birth'       => 'required|date_format:' . env('INPUT_DATE_FORMAT'),
                        'gender'              => 'required|in:Male,Female',
                        'father_name'         => 'required',
                        'mother_name'         => 'required',
                        'email'               => 'required|email|unique:users,email',
                        'contact_number'      => 'required|unique:users,mobile_no',
                        'residency'           => 'required|in:resident,nonresident',
                        'nid_number'          => 'required|digits_between:13,17',
                        'occupation_id'       => 'required|exists:gs_occupation,occupation_id',
                        'nationality'         => 'required|exists:gs_nationality,nationality_id',
                        'present_address'     => 'required',
                        'permanent_address'   => 'required',
                    ],
                    [
                              // 'investor_details_id.required'      => 'The investor id is required',
                        'investor_name.required'     => 'The investor name field is required',
                        'product_id.required'        => 'The product field is required',
                        'product_id.exists'          => 'The product field is invalid',
                        'date_of_birth.required'     => 'The investor date of birth field is required',
                        'date_of_birth.date_format'  => 'The investor date of birth field must be in '. env('INPUT_DATE_FORMAT') .' format',
                        'gender.required'            => 'The gender from field is required',
                        'gender.in'                  => 'The gender from field is invalid',
                        'father_name.required'       => 'The father name field is required',
                        'mother_name.required'       => 'The mother name field is required',
                        'email.required'             => 'The email field is required',
                        'email.email'                => 'The email field is not a valid email address',
                        'email.unique'               => 'The email is already registered',
                        'residency.required'         => 'The residency field is required',
                        'residency.in'               => 'The residency field is invalid',
                        'nid_number.required'        => 'The nid field is required',
                        'nid_number.digits_between'  => 'The nid number must be either 13 or 17 digit',
                        'occupation_id.required'     => 'The occupation field is required',
                        'occupation_id.exists'       => 'The occupation field is invalid',
                        'nationality.required'       => 'The nationality field is required',
                        'nationality.exists'         => 'The nationality field is invalid',
                        'present_address.required'   => 'The present address field is required',
                        'permanent_address.required' => 'The permanent address field is required',
                    ]
                );

                      // dd($validator->fails());
    
                if ($validator->fails()) {
                    return response()->json([
                        'status'  => 'error',
                        'message' => 'Form validation error',
                        'errors'  => $validator->errors()->messages(),
                    ], 400);
                }

                      // dd('OKAY');


                $investorDetail = InvestorDetail::where('investor_details_id', $request->investor_details_id)
                ->where('reg_is_complete' , '=', null)
                ->first();

                      // dd(is_null($investorDetail));

                if (is_null($investorDetail)) {
                    $investorDetail = new InvestorDetail();
                }

                
                $investorDetail->org_branch_id     = 41;
                $investorDetail->operation_mode_id = 1;
                $investorDetail->investor_name     = $request->investor_name;
                $investorDetail->template_id       = $request->product_id;
                $investorDetail->dob               = Carbon::createFromFormat(env('INPUT_DATE_FORMAT'), $request->date_of_birth)->format('Y-m-d H:i:s');
                $investorDetail->gender            = $request->gender;
                $investorDetail->father_name       = $request->father_name;
                $investorDetail->mother_name       = $request->mother_name;
                $investorDetail->email             = $request->email;
                $investorDetail->mobile            = $request->contact_number;
                $investorDetail->residency         = $request->residency;
                $investorDetail->nid               = $request->nid_number;
                $investorDetail->occupation_id     = $request->occupation_id;
                $investorDetail->nationality_id    = $request->nationality_id;
                $investorDetail->mailing_address   = $request->present_address;
                $investorDetail->permanent_address = $request->permanent_address;
                $investorDetail->msg_status        = 'pending';
                $investorDetail->reg_page_no       = 1;
                $investorDetail->business_date     = $businessDate->business_date;
                $investorDetail->record_date       = now();
                $investorDetail->save();

                return response()->json([
                    'status'              => 'success',
                    'investor_details_id' => $investorDetail->investor_details_id,
                ]);

            } elseif ($pageId === 4) {
                      /**
                 * insert/update nominee information
                 */
                $validator = Validator::make(
                    $request->all(),
                    [
                        'investor_details_id'     => 'required|exists:is_investor_details,investor_details_id',
                        'nominee_info_id'         => 'nullable',
                        'nominee_full_name'       => 'required',
                        'nominee_share'           => 'required|numeric',
                        'nominee_date_of_birth'   => 'required|date_format:' . env('INPUT_DATE_FORMAT'),
                        'nominee_father_name'     => 'required',
                        'nominee_mother_name'     => 'required',
                        'nominee_contact_number'  => 'required',
                        'nominee_nid'             => 'required',
                        'nominee_occupation_id'   => 'exists:gs_occupation,occupation_id',
                        'nominee_relationship_id' => 'required|exists:gs_relation,relation_id',
                    ],
                    [
                        'investor_details_id.required'      => "The investor details id is required",
                        'investor_details_id.exists'        => "The investor details id is invalid",
                        'nominee_full_name.required'        => "The nominee full name field is required",
                        'nominee_share.required'            => "The nominee share field is required",
                        'nominee_share.numeric'             => "The nominee share field must be a number",
                        'nominee_date_of_birth.required'    => "The nominee date of birth field is required",
                        'nominee_date_of_birth.date_format' => "The nominee date of birth field must be in " . env('INPUT_DATE_FORMAT') . " format",
                        'nominee_father_name.required'      => "The nominee father name field is required",
                        'nominee_mother_name.required'      => "The nominee mother name field is required",
                        'nominee_contact_number.required'   => "The nominee contact number field is required",
                        'nominee_nid.required'              => "The nominee nid field is required",
                        'nominee_occupation_id.required'    => "The nominee occupation field is required",
                        'nominee_occupation_id.exists'      => "The nominee occupation field is invalid",
                        'nominee_relationship_id.required'  => "The nominee relationship field is required",
                        'nominee_relationship_id.exists'    => "The nominee relationship field is invalid",
                    ]
                );

    
                if ($validator->fails()) {
                    return response()->json([
                        'status'  => 'error',
                        'message' => 'Form validation error',
                        'errors'  => $validator->errors()->messages(),
                    ], 400);
                }

                
                $nomineeInfo = NomineeInfo::where('investor_details_id', $request->investor_details_id)->first();

                      // dd(is_null($nomineeInfo));

                if (is_null($nomineeInfo)) {
                    $nomineeInfo = new NomineeInfo();
                }

                $nomineeInfo->investor_details_id = $request->investor_details_id;
                $nomineeInfo->nominee_name        = $request->nominee_full_name;
                $nomineeInfo->share_percentage    = $request->nominee_share;
                $nomineeInfo->dob                 = Carbon::createFromFormat(env('INPUT_DATE_FORMAT'), $request->nominee_date_of_birth)->format('Y-m-d H:i:s');
                $nomineeInfo->father_name         = $request->nominee_father_name;
                $nomineeInfo->mother_name         = $request->nominee_mother_name;
                $nomineeInfo->spouse_name         = $request->nominee_spouse_name;
                $nomineeInfo->nominee_mobile      = $request->nominee_contact_number;
                $nomineeInfo->nid                 = $request->nominee_nid;
                $nomineeInfo->mailing_address     = $request->nominee_mailing_address;
                $nomineeInfo->occupation_id       = $request->nominee_occupation_id;
                $nomineeInfo->relation_id         = $request->nominee_relationship_id;
                $nomineeInfo->business_date       = $businessDate->business_date;
                $nomineeInfo->record_date         = now();
                $nomineeInfo->save();


                $investorDetail              = InvestorDetail::find($request->investor_details_id);
                $investorDetail->reg_page_no = 2;
                $investorDetail->save();

                return response()->json([
                    'status'              => 'success',
                    'investor_details_id' => $request->investor_details_id,
                    'nominee_info_id'     => $nomineeInfo->nominee_info_id,
                ]);


            } elseif ($pageId === 5) {
                      /**
                 * insert deposit information
                 */
                
                $validator = Validator::make(
                    $request->all(),
                    [
                        'investor_details_id'   => 'required|exists:is_investor_details,investor_details_id',
                        'deposit_amount'        => 'required|numeric',
                        'deposit_account_id'    => 'required|exists:gs_account_details,account_details_id',
                        'payment_type'          => 'required|in:cheque,po,dd,cd,ft',
                        'deposit_instrument_no' => 'required',
                        'bank_id'               => 'required|exists:gs_money_mkt_org,org_id',
                        'deposit_date'          => 'required|date_format:' . env('INPUT_DATE_FORMAT'),
                        'deposit_slip'          => 'required|file|extensions:jpg,png,jpeg,pdf',
                    ],
                    [
                        'investor_details_id.required'      => "The investor details id is required",
                        'investor_details_id.exists'        => "The investor details id is invalid",
                        'deposit_amount.required'        => 'The deposit amount filed is required',
                        'deposit_amount.numeric'         => 'The deposit amount must be a number',
                        'deposit_account_id.required'    => 'The deposit account filed name is required',
                        'deposit_account_id.exists'      => 'The deposit account is invalid',
                        'payment_type.required'          => 'The payment type filed is required',
                        'payment_type.in'                => 'The payment type is invalid',
                        'deposit_instrument_no.required' => 'The deposit instrument number filed is required',
                        'bank_id.required'               => 'The bank name filed is required',
                        'bank_id.exists'                 => 'The bank name is invalid',
                        'deposit_date.required'          => 'The deposit instrument date filed is required',
                        'deposit_date.date_format'       => 'The deposit instrument date field must be in ' . env('INPUT_DATE_FORMAT') . '  format',
                        'deposit_slip.required'          => 'The deposit slip filed is required',
                        'deposit_slip.file'              => 'The deposit slip must be a file',
                        'deposit_slip.extensions'        => 'The deposit slip must be in jpg, png, jpeg or pdf format',
                    ]
                );
        
                if ($validator->fails()) {
                    return response()->json([
                        'status'  => 'error',
                        'message' => 'Form validation error',
                        'errors'  => $validator->errors()->messages(),
                    ], 400);
                }
                
                try {
                    $depositSlipImageName = null;
                    $businessDate         = BusinessDate::latest('business_date')->first();
    
                    if ($request->hasFile('deposit_slip')) {
                        $depositSlipImage     = $request->file('deposit_slip');
                        $depositSlipImageName = 'deposit_slip_'.$request->investor_details_id.'_'.time().'.'.$depositSlipImage->getClientOriginalExtension();
                        $depositSlipImage->move(public_path().'/upload/deposit_slip', $depositSlipImageName);    
                    }

                    $transactionDetails = TransactionDetails::where('investor_details_id', $request->investor_details_id)->first();

                    if (is_null($transactionDetails)) {
                        $transactionDetails                        = new TransactionDetails();
                    }
    
                    $transactionDetails->org_id                = $request->bank_id;
                    $transactionDetails->investor_details_id   = $request->investor_details_id;
                    $transactionDetails->transaction_type      = 'deposit';
                    $transactionDetails->payment_type          = $request->payment_type;
                    $transactionDetails->chq_no                = $request->deposit_instrument_no;
                    $transactionDetails->chq_date              = Carbon::createFromFormat(env('INPUT_DATE_FORMAT'), $request->deposit_date)->format('Y-m-d H:i:s');
                    $transactionDetails->amount                = $request->deposit_amount;
                    $transactionDetails->remarks               = 'web deposit';
                    $transactionDetails->operate_by            = 0;
                    $transactionDetails->msg_status            = '';
                    $transactionDetails->account_details_id    = $request->deposit_account_id;
                    $transactionDetails->file_upload           = $depositSlipImageName;
                    $transactionDetails->send_to_bank          = 'n';
                    $transactionDetails->honor_dishonor_cancel = 'p';
                    $transactionDetails->cheque_print_flag     = 'n';
                    $transactionDetails->business_date         = $businessDate->business_date;
                    $transactionDetails->record_date           = now();
    
                    if ($transactionDetails->save()) {
                        $messageRoute                         = new MessageRoute();
                        $messageRoute->send_by                = '4772';
                        $messageRoute->receive_by             = '4772';
                        $messageRoute->viewableto             = 4;
                        $messageRoute->transaction_details_id = $transactionDetails->transaction_details_id;
                        $messageRoute->remarks                = 'Deposit by Investor';
                        $messageRoute->status                 = '0';
                        $messageRoute->business_date          = $businessDate->business_date;
                        $messageRoute->record_date            = now();
                        $messageRoute->send_date              = now();
    
                        $messageRoute->save();
    
                        return response()->json([
                            'status'  => 'success',
                            'message' => 'Deposit submitted successfully',
                        ]);
                    }
                } catch (\Throwable $th) {
                    return response()->json([
                        'status'      => 'error',
                        'message'     => 'Deposit form submit unsuccessful',
                        'dev_message' => $th->getMessage(),
                    ], 400);
                }


            } elseif ($pageId === 6) {
                      /**
                 * insert file for investor and nominee
                 */
                $validator = Validator::make(
                    $request->all(),
                    [
                        'investor_details_id'   => 'required|exists:is_investor_details,investor_details_id',
                        'nominee_info_id'       => 'required|exists:is_nominee_info,nominee_info_id',
                        'investor_photo'        => 'required|file|extensions:jpg,png,jpeg',
                        'investor_nid_photo'    => 'required|file|extensions:jpg,png,jpeg',
                        'investor_signature'    => 'required|file|extensions:jpg,png,jpeg',
                        'investor_cheque_photo' => 'required|file|extensions:jpg,png,jpeg',
                        'nominee_photo'         => 'required|file|extensions:jpg,png,jpeg',
                        'nominee_nid_photo'     => 'required|file|extensions:jpg,png,jpeg',
                        'nominee_signature'     => 'required|file|extensions:jpg,png,jpeg',
                    ],
                    [
                        'investor_details_id.required'     => "The investor details id is required",
                        'investor_details_id.exists'       => "The investor details id is invalid",
                        'nominee_info_id.required'         => "The nominee info id is required",
                        'nominee_info_id.exists'           => "The nominee info id is invalid",
                        'investor_photo.required'          => 'The investor photo filed is required',
                        'investor_photo.file'              => 'The investor photo must be a file',
                        'investor_photo.extensions'        => 'The investor photo must be in jpg, png or jpeg format',
                        'investor_nid_photo.required'      => 'The investor nid photo filed is required',
                        'investor_nid_photo.file'          => 'The investor nid photo must be a file',
                        'investor_nid_photo.extensions'    => 'The investor nid photo must be in jpg, png or jpeg format',
                        'investor_signature.required'      => 'The investor signature filed is required',
                        'investor_signature.file'          => 'The investor signature must be a file',
                        'investor_signature.extensions'    => 'The investor signature must be in jpg, png or jpeg format',
                        'investor_cheque_photo.required'   => 'The investor cheque photo filed is required',
                        'investor_cheque_photo.file'       => 'The investor cheque photo must be a file',
                        'investor_cheque_photo.extensions' => 'The investor cheque photo must be in jpg, png or jpeg format',
                        'nominee_photo.required'           => 'The nominee photo filed is required',
                        'nominee_photo.file'               => 'The nominee photo must be a file',
                        'nominee_photo.extensions'         => 'The nominee photo must be in jpg, png or jpeg format',
                        'nominee_nid_photo.required'       => 'The nominee nid photo filed is required',
                        'nominee_nid_photo.file'           => 'The nominee nid photo must be a file',
                        'nominee_nid_photo.extensions'     => 'The nominee nid photo must be in jpg, png or jpeg format',
                        'nominee_signature.required'       => 'The nominee signature filed is required',
                        'nominee_signature.file'           => 'The nominee signature must be a file',
                        'nominee_signature.extensions'     => 'The nominee signature must be in jpg, png or jpeg format',
                    ]
                );
                

                if ($validator->fails()) {
                    return response()->json([
                        'status'  => 'error',
                        'message' => 'Form validation error',
                        'errors'  => $validator->errors()->messages(),
                    ], 400);
                }

                $investorImageName          = null;
                $investorNidImageName       = null;
                $investorSignatureImageName = null;
                $investorChequeImageName    = null;
                $nomineeImageName           = null;
                $nomineeNidImageName        = null;
                $nomineeSignatureImageName  = null;

                if ($request->hasFile('investor_photo')) {
                    $investorImage     = $request->file('investor_photo');
                    $investorImageName = 'investor_photo'.$request->investor_details_id.'_'.time().'.'.$investorImage->getClientOriginalExtension();
                    $investorImage->move(public_path().'/upload/investor_photo', $investorImageName);
                }

                if ($request->hasFile('investor_nid_photo')) {
                    $investorNidImage     = $request->file('investor_nid_photo');
                    $investorNidImageName = 'investor_nid_photo'.$request->investor_details_id.'_'.time().'.'.$investorNidImage->getClientOriginalExtension();
                    $investorNidImage->move(public_path().'/upload/investor_nid_photo', $investorNidImageName);
                }

                if ($request->hasFile('investor_signature')) {
                    $investorSignatureImage     = $request->file('investor_signature');
                    $investorSignatureImageName = 'investor_signature'.$request->investor_details_id.'_'.time().'.'.$investorSignatureImage->getClientOriginalExtension();
                    $investorSignatureImage->move(public_path().'/upload/investor_signature', $investorSignatureImageName);
                }

                if ($request->hasFile('investor_cheque_photo')) {
                    $investorChequeImage     = $request->file('investor_cheque_photo');
                    $investorChequeImageName = 'investor_cheque_photo'.$request->investor_details_id.'_'.time().'.'.$investorChequeImage->getClientOriginalExtension();
                    $investorChequeImage->move(public_path().'/upload/investor_cheque_photo', $investorChequeImageName);
                }

                if ($request->hasFile('nominee_photo')) {
                    $nomineeImage     = $request->file('nominee_photo');
                    $nomineeImageName = 'nominee_photo'.$request->investor_details_id.'_'.time().'.'.$nomineeImage->getClientOriginalExtension();
                    $nomineeImage->move(public_path().'/upload/nominee_photo', $nomineeImageName);
                }

                if ($request->hasFile('nominee_nid_photo')) {
                    $nomineeNidImage     = $request->file('nominee_nid_photo');
                    $nomineeNidImageName = 'nominee_nid_photo'.$request->investor_details_id.'_'.time().'.'.$nomineeNidImage->getClientOriginalExtension();
                    $nomineeNidImage->move(public_path().'/upload/nominee_nid_photo', $nomineeNidImageName);
                }

                if ($request->hasFile('nominee_signature')) {
                    $nomineeSignatureImage     = $request->file('nominee_signature');
                    $nomineeSignatureImageName = 'nominee_signature'.$request->investor_details_id.'_'.time().'.'.$nomineeSignatureImage->getClientOriginalExtension();
                    $nomineeSignatureImage->move(public_path().'/upload/nominee_signature', $nomineeSignatureImageName);
                }

                $nomineeInfo = NomineeInfo::where('nominee_info_id', $request->nominee_info_id)
                ->where('investor_details_id', $request->investor_details_id)
                ->first();

                $investorDetail = InvestorDetail::find($request->investor_details_id);

                if (is_null($nomineeInfo) || is_null($investorDetail)) {
                    return response()->json([
                        'status'  => 'error',
                        'message' => 'Form validation error',
                    ], 400);
                }

                $nomineeInfo->nominee_applicant_photo = $nomineeImageName;
                $nomineeInfo->nominee_nid             = $nomineeNidImageName;
                $nomineeInfo->nominee_applicant_sign  = $nomineeSignatureImageName;
                $nomineeInfo->save();

                $investorDetail->applicant_photo = $investorImageName;
                $investorDetail->nid_copy        = $investorNidImageName;
                $investorDetail->applicant_sign  = $investorSignatureImageName;
                $investorDetail->bank_copy       = $investorChequeImageName;
                $investorDetail->reg_page_no     = 3;
                $investorDetail->reg_is_complete = 'yes';
                $investorDetail->save();
                

                return response()->json([
                    'status'              => 'success',
                    'message'             => 'Registration successful',
                    'investor_details_id' => $request->investor_details_id,
                    'nominee_info_id'     => $nomineeInfo->nominee_info_id,
                ]);

            } else {
                

                      // $validator = Validator::make(
                      //     $request->all(),
                      //     [
                      //         'investor_name'           => 'required',
                      //         'product_id'              => "required|exists:ls_template,template_id",
                      //         'date_of_birth'           => 'required|date_format:d-m-Y',
                      //         'gender'                  => 'required|in:Male,Female',
                      //         'father_name'             => 'required',
                      //         'mother_name'             => 'required',
                      //         'email'                   => 'required|email|unique:users,email',
                      //         'contact_number'          => 'required|unique:users,mobile_no',
                      //         'residency'               => 'required|in:resident,nonresident',
                      //         'nid_number'              => 'required|digit:13,17',
                      //         'occupation_id'           => 'required|exists:gs_occupation,occupation_id',
                      //         'nationality'             => 'required|exists:gs_nationality,nationality_id',
                      //         'present_address'         => 'required',
                      //         'permanent_address'       => 'required',
                      //         'nominee_full_name'       => 'required',
                      //         'nominee_share'           => 'required|numeric',
                      //         'nominee_date_of_birth'   => 'required|date_format:d-m-Y',
                      //         'nominee_father_name'     => 'required',
                      //         'nominee_mother_name'     => 'required',
                      //         'nominee_contact_number'  => 'required',
                      //         'nominee_nid'             => 'required',
                      //         'nominee_occupation_id'   => 'exists:gs_occupation,occupation_id',
                      //         'nominee_relationship_id' => 'required|exists:gs_relation,relation_id',
                      //         'investor_photo'          => 'required|file|extensions:jpg,png,jpeg',
                      //         'investor_nid_photo'      => 'required|file|extensions:jpg,png,jpeg',
                      //         'investor_signature'      => 'required|file|extensions:jpg,png,jpeg',
                      //         'investor_cheque_photo'   => 'required|file|extensions:jpg,png,jpeg',
                      //         'nominee_photo'           => 'required|file|extensions:jpg,png,jpeg',
                      //         'nominee_nid_photo'       => 'required|file|extensions:jpg,png,jpeg',
                      //         'nominee_signature'       => 'required|file|extensions:jpg,png,jpeg',
                      //     ],
                      //     [
                      //         'investor_name.required'            => 'The investor name field is required',
                      //         'product_id.required'               => 'The product field is required',
                      //         'product_id.exists'                 => 'The product field is invalid',
                      //         'date_of_birth.required'            => "The investor date of birth field is required",
                      //         'date_of_birth.date_format'         => "The investor date of birth field must be in d-m-Y format",
                      //         'gender.required'                   => "The gender from field is required",
                      //         'gender.in'                         => "The gender from field is invalid",
                      //         'father_name.required'              => "The father name field is required",
                      //         'mother_name.required'              => "The mother name field is required",
                      //         'email.required'                    => "The email field is required",
                      //         'email.email'                       => "The email field is not a valid email address",
                      //         'email.unique'                      => "The email is already registered",
                      //         'residency.required'                => "The residency field is required",
                      //         'residency.in'                      => "The residency field is invalid",
                      //         'nid_number.required'               => "The nid field is required",
                      //         'nid_number.digit'                  => "The nid number must be either 13 or 17 digit",
                      //         'occupation_id.required'            => "The occupation field is required",
                      //         'occupation_id.exists'              => "The occupation field is invalid",
                      //         'nationality.required'              => "The nationality field is required",
                      //         'nationality.exists'                => "The nationality field is invalid",
                      //         'present_address.required'          => "The present address field is required",
                      //         'permanent_address.required'        => "The permanent address field is required",
                      //         'nominee_full_name.required'        => "The nominee full name field is required",
                      //         'nominee_share.required'            => "The nominee share field is required",
                      //         'nominee_share.numeric'             => "The nominee share field must be a number",
                      //         'nominee_date_of_birth.required'    => "The nominee date of birth field is required",
                      //         'nominee_date_of_birth.date_format' => "The nominee date of birth field must be in d-m-Y format",
                      //         'nominee_father_name.required'      => "The nominee father name field is required",
                      //         'nominee_mother_name.required'      => "The nominee mother name field is required",
                      //         'nominee_contact_number.required'   => "The nominee contact number field is required",
                      //         'nominee_nid.required'              => "The nominee nid field is required",
                      //         'nominee_occupation_id.required'    => "The nominee occupation field is required",
                      //         'nominee_occupation_id.exists'      => "The nominee occupation field is invalid",
                      //         'nominee_relationship_id.required'  => "The nominee relationship field is required",
                      //         'nominee_relationship_id.exists'    => "The nominee relationship field is invalid",
                      //         'investor_photo.required'           => 'The investor photo filed is required',
                      //         'investor_photo.file'               => 'The investor photo must be a file',
                      //         'investor_photo.extensions'         => 'The investor photo must be in jpg, png or jpeg format',
                      //         'investor_nid_photo.required'       => 'The investor nid photo filed is required',
                      //         'investor_nid_photo.file'           => 'The investor nid photo must be a file',
                      //         'investor_nid_photo.extensions'     => 'The investor nid photo must be in jpg, png or jpeg format',
                      //         'investor_signature.required'       => 'The investor signature filed is required',
                      //         'investor_signature.file'           => 'The investor signature must be a file',
                      //         'investor_signature.extensions'     => 'The investor signature must be in jpg, png or jpeg format',
                      //         'investor_cheque_photo.required'    => 'The investor cheque photo filed is required',
                      //         'investor_cheque_photo.file'        => 'The investor cheque photo must be a file',
                      //         'investor_cheque_photo.extensions'  => 'The investor cheque photo must be in jpg, png or jpeg format',
                      //         'nominee_photo.required'            => 'The nominee photo filed is required',
                      //         'nominee_photo.file'                => 'The nominee photo must be a file',
                      //         'nominee_photo.extensions'          => 'The nominee photo must be in jpg, png or jpeg format',
                      //         'nominee_nid_photo.required'        => 'The nominee nid photo filed is required',
                      //         'nominee_nid_photo.file'            => 'The nominee nid photo must be a file',
                      //         'nominee_nid_photo.extensions'      => 'The nominee nid photo must be in jpg, png or jpeg format',
                      //         'nominee_signature.required'        => 'The nominee signature filed is required',
                      //         'nominee_signature.file'            => 'The nominee signature must be a file',
                      //         'nominee_signature.extensions'      => 'The nominee signature must be in jpg, png or jpeg format',
                      //     ]
                      // );
            }
        } catch (\Throwable $th) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Form submission error.',
                      'dev_message' => $th->getMessage(),
                      'file' => $th->getFile(),
                      'line' => $th->getLine(),
            ], 400);
        }
    }

    public function changePassword(Request $request)
    {
        $validator = Validator::make(
            $request->all(),
            [
                'current_password'          => 'required',
                'new_password'              => 'required|string|min:8|confirmed',
                'new_password_confirmation' => 'required',
            ],
            [
                'current_password.required' => 'The current password field is required',
                'new_password.required'     => 'The new password field is required',
                'new_password.min'          => 'The new password field must be 8 char long',
                'new_password.confirmed'    => 'The new password must be match with confirmed password',
            ]
        );

        if ($validator->fails()) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Form validation error',
                'errors'  => $validator->errors()->messages(),
            ], 400);
        }

        auth()->user()->update( [
                'password' => Hash::make($request->new_password)
        ]);

        return response()->json([
            'status'  => 'success',
            'message' => 'Password has been changed'
        ]);
    }

    public function signUp(Request $request, int $pageId)
    {
        try {
            if ($pageId === 1) {
                      /**
                 * Find the investor with email or mobile or portfolio code
                 */
                $validator = Validator::make(
                    $request->all(),
                    [
                        'email'          => 'required|email',
                        'contact_number' => 'required|digits:11',
                        'portfolio_code' => 'required',
                    ],
                    [
                        'email.required'          => 'The email field is required',
                        'email.email'             => 'The email field is not a valid email address',
                        'contact_number.required' => 'The contact number field is required',
                        'contact_number.digits'   => 'The contact number field must be 11 digits',
                        'portfolio_code.required' => 'The portfolio code field is required',
                    ]
                );

                if ($validator->fails()) {
                    return response()->json([
                        'status'  => 'error',
                        'message' => 'Form validation error',
                        'errors'  => $validator->errors()->messages(),
                    ], 400);
                } else {
                    $investorDetail = InvestorDetail::where('email', $request->email)
                    ->where('mobile', $request->contact_number)
                    ->where('portfolio_code', $request->portfolio_code)
                    ->first();

                    if (!is_null($investorDetail)) {
                        $user = User::where('email', $request->email)
                        ->orWhere('mobile_no', $request->contact_number)
                        ->orWhere('investor_code', $request->portfolio_code)
                        ->first();

                        if (is_null($user)) {

                            $otpSend = true;
                            if ($otpSend) {
                                return response()->json([
                                    'status'  => 'success',
                                    'message' => 'Your OTP has been sent to you email or mobile',
                                ]);
                            }

                            return response()->json([
                                'status'  => 'error',
                                'message' => 'We are unable to send otp to your email or mobile',
                            ], 401);
                        }
                        
                        return response()->json([
                            'status'  => 'error',
                            'message' => 'An investor already registered with providing details',
                        ], 400);
                    }
                        
                    return response()->json([
                        'status'  => 'error',
                        'message' => 'We could not found you details in our system with providing details',
                    ], 400);
                }
            } elseif ($pageId === 2) {
                      /**
                 * verify otp
                 */
                $validator = Validator::make(
                    $request->all(),
                    [
                        'otp_code' => 'required|numeric|digits:6',
                    ],
                    [
                        'otp_code.required' => 'The otp field is required',
                        'otp_code.numeric'  => 'The otp field must be a number',
                        'otp_code.digits'   => 'The otp field must be 6 digits',
                    ]
                );
        
                if ($validator->fails()) {
                    return response()->json([
                        'status'  => 'error',
                        'message' => 'Form validation error',
                        'errors'  => $validator->errors()->messages(),
                    ], 400);
                }
        
                if (true) {
                    return response()->json([
                        'status'     => 'success',
                        'otp_verify' => true,
                    ]);
                }
        
                return response()->json([
                    'status'     => 'error',
                    'otp_verify' => false,
                ]);
            } elseif ($pageId === 3) {
                      /**
                 * Insert/update investor details
                 */
                $validator = Validator::make(
                    $request->all(),
                    [
                        'email'                 => 'required|email',
                        'contact_number'        => 'required|digits:11',
                        'portfolio_code'        => 'required',
                        'password'              => 'required|digits:8|confirmed',
                        'password_confirmation' => 'required',
                    ],
                    [
                        'email.required'                 => 'The email field is required',
                        'email.email'                    => 'The email field is not a valid email address',
                        'contact_number.required'        => 'The contact number field is required',
                        'contact_number.digits'          => 'The contact number field must be 11 digits',
                        'portfolio_code.required'        => 'The portfolio code field is required',
                        'password.required'              => 'The password field is required',
                        'password.digits'                => 'The password field must be 8 digits',
                        'password.confirmed'             => 'The password field must be matched with confirm password field',
                        'password_confirmation.required' => 'The confirm password field is required',
                    ]
                );

                      // dd($validator->fails());
    
                if ($validator->fails()) {
                    return response()->json([
                        'status'  => 'error',
                        'message' => 'Form validation error',
                        'errors'  => $validator->errors()->messages(),
                    ], 400);
                }

                $investorDetail = InvestorDetail::where('portfolio_code', $request->portfolio_code)->first();
                
                $user                = new User();
                $user->name          = $investorDetail->investor_name;
                $user->investor_code = $request->portfolio_code;
                $user->email         = $request->email;
                $user->password      = Hash::make($request->password);
                $user->mobile_no     = $request->contact_number;
                $user->created_at    = now();
                $user->updated_at    = now();
                $user->save();

                return response()->json([
                    'status'  => 'success',
                    'message' => 'Sign up successfull',
                ]);
            } else {
                return response()->json([
                    'status'  => 'success',
                    'message' => 'Unknown request',
                ], 400);
            }
            
        } catch (\Throwable $th) {
            return response()->json([
                'status'      => 'error',
                'dev_message' => $th->getMessage(),
                'file'        => $th->getFile(),
                'line'        => $th->getLine(),
                'message'     => 'Deposit form submit unsuccessful',
            ], 400);
        }        
    }

    public function forgotPassword(Request $request, int $pageId)
    {
        try {
            if ($pageId === 1) {

                // dd($pageId);

                $validator = Validator::make(
                    $request->all(),
                    [
                        'email_or_mobile'              => 'required',
                    ],
                    [
                        'email_or_mobile.required' => 'The email or mobile field is required',
                    ]
                );

                // dd($validator->fails());
        
                if ($validator->fails()) {
                    return response()->json([
                        'status'  => 'error',
                        'message' => 'Form validation error',
                        'errors'  => $validator->errors()->messages(),
                    ], 400);
                }
                
                $user = User::where('email', $request->email_or_mobile)
                ->orWhere('mobile_no', $request->email_or_mobile)
                ->exclude(['password'])
                ->first();

                // $user = User::all();

                // dd($user);
        
                if (is_null($user)) {
                    return response()->json([
                        'status'  => 'error',
                        'message' => 'We cound not found any investor with providing email or mobile'
                    ], 400);
                }

                // if otp send success
                $otpSend = true;
                if ($otpSend) {
                    return response()->json([
                        'status'  => 'success',
                        'user'          => $user,
                        'message' => 'Your OTP has been sent to you email or mobile',
                        // 'message' => 'An otp already send to your mobile or email',
                    ]);
                }
        
                return response()->json([
                    'status'  => 'error',
                    'message' => 'We are unable to send otp to your email or mobile'
                ], 401);
            }
            
            if ($pageId === 2) {
                /**
                 * verify otp
                 */
                $validator = Validator::make(
                    $request->all(),
                    [
                        'otp_code' => 'required|numeric|digits:6',
                    ],
                    [
                        'otp_code.required' => 'The otp field is required',
                        'otp_code.numeric'  => 'The otp field must be a number',
                        'otp_code.digits'   => 'The otp field must be 6 digits',
                    ]
                );
        
                if ($validator->fails()) {
                    return response()->json([
                        'status'  => 'error',
                        'message' => 'Form validation error',
                        'errors'  => $validator->errors()->messages(),
                    ], 400);
                }
        
                // if otp verify success
                $otpVerify = true;
                if ($otpVerify) {
                    return response()->json([
                        'status'     => 'success',
                        'otp_verify' => true,
                    ]);
                }
        
                return response()->json([
                    'status'     => 'error',
                    'otp_verify' => false,
                ]);
            }

            if ($pageId === 3) {
                /**
                 * change password
                 */
                $validator = Validator::make(
                    $request->all(),
                    [
                        'user_id'                   => 'required|exists:users,id',
                        'new_password'              => 'required|string|min:8|confirmed',
                        'new_password_confirmation' => 'required',
                    ],
                    [
                        'user_id.required'          => 'The user id is missing',
                        'user_id.exists'            => 'The user id is invalid',
                        'new_password.required'     => 'The new password field is required',
                        'new_password.min'          => 'The new password field must be 8 char long',
                        'new_password.confirmed'    => 'The new password must be match with confirmed password',
                    ]
                );
        
                if ($validator->fails()) {
                    return response()->json([
                        'status'  => 'error',
                        'message' => 'Form validation error',
                        'errors'  => $validator->errors()->messages(),
                    ], 400);
                }

                User::find($request->user_id)->update([
                        'password' => Hash::make($request->new_password)
                ]);
        
                return response()->json([
                    'status'  => 'success',
                    'message' => 'Password has been changed'
                ]);
            }
        } catch (\Throwable $th) {
            return response()->json([
                'status'  => 'error',
                'dev_message' => $th->getMessage(),
                'message' => 'Form validation error',
            ], 400);
        }
    }
}
