<?php

namespace App\Http\Controllers\Report;

use Carbon\Carbon;
use Illuminate\Http\Request;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class DepositStatementController extends Controller
{
    public function depositStatementDownload(Request $request)
    {
        $validator = Validator::make(
            $request->all(),
            [
                'from_date'  => 'required|date_format:Y-m-d',
                'to_date'  => 'required|date_format:Y-m-d',
            ],
            [
                'from_date.required'          => 'The as on date filed is required',
                'from_date.date_format'       => 'The as on date field must be in Y-m-d format',
                'to_date.required'          => 'The as on date filed is required',
                'to_date.date_format'       => 'The as on date field must be in Y-m-d format',
            ]
        );

        if ($validator->fails()) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Form validation error',
                'errors' => $validator->errors()->messages(),
            ], 400);
        }

        $data = [
            'title' => 'Deposit Statement',
        ];

		$user                    = Auth::user();
		$investorDetail          = $user->investorDetail;		
        $fromDate = Carbon::createFromFormat('Y-m-d', $request->from_date);
		$toDate   = Carbon::createFromFormat('Y-m-d', $request->to_date);
		$data['account_type']    = $investorDetail->templateDetails->product_short_name;
		$data['account_no']      = $investorDetail->portfolio_code;
		$data['bo_no']           = $investorDetail->bo_account_no;
		$data['account_date']    = Carbon::createFromFormat('Y-m-d H:i:s', $investorDetail->business_date)->format(env('OUTPUT_DATE_FORMAT'));
		$data['account_name']    = $investorDetail->investor_name;
		$data['account_address'] = $investorDetail->mailing_address;
        $data['report_date']     = $fromDate->format('d-m-Y') . ' To ' . $toDate->format('d-m-Y');

        if ($investorDetail->account_enable_disable == 'e') {
			$data['account_status'] = 'Active';
		}

		if ($investorDetail->account_close == 'y') {
			$data['account_status'] = 'Close';
		}

		if ($investorDetail->account_suspend == 'y') {
			$data['account_status'] = 'Suspend';
		}



        $fundDepositSql = "
            SELECT 
                DISTINCT TD.TRANSACTION_DETAILS_ID, 
                TD.PORTFOLIO_CODE AS PORTFOLIO_CODE, 
                COALESCE(
                    INVD.INSTITUTION_NAME, INVD.INVESTOR_NAME
                ) AS INVESTOR_NAME, 
                TD.PAYMENT_TYPE, 
                TD.CHQ_NO, 
                TD.CHQ_DATE, 
                TD.AMOUNT, 
                TD.REMARKS, 
                TD.BUSINESS_DATE AS BUSINESS_DATE1, 
                TD.HONOR_DISHONOR_CANCEL_DATE AS BUSINESS_DATE 
            FROM 
                IS_TRANSACTION_DETAILS TD, 
                IS_INVESTOR_DETAILS INVD 
            WHERE 
                TD.PORTFOLIO_CODE = INVD.PORTFOLIO_CODE 
                AND LOWER(TD.TRANSACTION_TYPE) = 'deposit' 
                AND LOWER(TD.SEND_TO_BANK) = 'y' 
                AND LOWER(TD.HONOR_DISHONOR_CANCEL) = 'h' 
                AND INVD.IS_TEMPLATE = 'n' 
                AND TD.AMOUNT > 0 
                AND INVD.PORTFOLIO_CODE = :portfolio_code
                AND TD.HONOR_DISHONOR_CANCEL_DATE BETWEEN :from_date AND :to_date
                ORDER BY 
                TD.BUSINESS_DATE ASC

        ";

        $fundDepositResult = DB::select(
            $fundDepositSql,
            [
                'from_date' => $fromDate->format('Y-m-d'),
                'to_date' => $toDate->format('Y-m-d'),
                'portfolio_code' => $user->investor_code,
            ]
        );

        // dd($fundDepositResult);

        $data['fundDeposit'] = $fundDepositResult;

        // return view('deposit.index', $data);

        $reportName = 'DepositStatement_' . $fromDate->format('d_m_Y') . '_' . $toDate->format('d_m_Y') . '.pdf';
        $pdf = Pdf::loadView('deposit.index', $data);
        $pdf->setPaper('a4');
        return $pdf->save($reportName)->stream($reportName);
    }
}
