<?php

namespace App\Http\Controllers\Report;

use Carbon\Carbon;
use Illuminate\Http\Request;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class GainLossSummuryController extends Controller
{
    public function gainLossDownload(Request $request)
    {
        $validator = Validator::make(
            $request->all(),
            [
                'from_date'  => 'required|date_format:Y-m-d',
                'to_date'  => 'required|date_format:Y-m-d',
            ],
            [
                'from_date.required'          => 'The as on date filed is required',
                'from_date.date_format'       => 'The as on date field must be in Y-m-d format',
                'to_date.required'          => 'The as on date filed is required',
                'to_date.date_format'       => 'The as on date field must be in Y-m-d format',
            ]
        );

        if ($validator->fails()) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Form validation error',
                'errors' => $validator->errors()->messages(),
            ], 400);
        }

        $data = [
            'title' => 'Capital Gain/Loss (Summary)',
        ];

        $user                    = Auth::user();
		$investorDetail          = $user->investorDetail;		
		$toDate                  = Carbon::createFromFormat('Y-m-d', $request->to_date);
        $fromDate                = Carbon::createFromFormat('Y-m-d', $request->from_date);
		$data['account_type']    = $investorDetail->templateDetails->product_short_name;
		$data['account_no']      = $investorDetail->portfolio_code;
		$data['bo_no']           = $investorDetail->bo_account_no;
		$data['account_date']    = Carbon::createFromFormat('Y-m-d H:i:s', $investorDetail->business_date)->format(env('OUTPUT_DATE_FORMAT'));
		$data['account_name']    = $investorDetail->investor_name;
		$data['account_address'] = $investorDetail->mailing_address;
        $data['report_date']     = $fromDate->format('d-m-Y') . ' To ' . $toDate->format('d-m-Y');

        if ($investorDetail->account_enable_disable == 'e') {
			$data['account_status'] = 'Active';
		}

		if ($investorDetail->account_close == 'y') {
			$data['account_status'] = 'Close';
		}

		if ($investorDetail->account_suspend == 'y') {
			$data['account_status'] = 'Suspend';
		}

        // dd($investorDetail);

        $gainLossData      = $this->getGainLossSummury($user->investor_code, $fromDate, $toDate);
        // gainloss

        $data['gainLossData'] = $gainLossData;
        $reportName = 'GainLossSummury_' . $fromDate->format('d_m_Y') . '_' . $toDate->format('d_m_Y') . '.pdf';
        $pdf = Pdf::loadView('gainloss.index', $data);
        $pdf->setPaper('a4');
        return $pdf->save($reportName)->stream($reportName);
    }

    private function getGainLossSummury($portfolioCode, $fromDate, $toDate)
    {
        $gainLossSql = "
            SELECT 
                CAPITAL_HAIN_DETAILS.PORTFOLIO_CODE, 
                CAPITAL_HAIN_DETAILS.INVESTOR_NAME, 
                CAPITAL_HAIN_DETAILS.INSTRUMENT_DETAILS_ID, 
                CAPITAL_HAIN_DETAILS.INSTRUMENT_NAME, 
                SUM (
                    CAPITAL_HAIN_DETAILS.NO_OF_SHARES
                ) AS NO_OF_SHARES, 
                (
                    SUM (
                    CAPITAL_HAIN_DETAILS.SALE_AMOUNT
                    ) / SUM (
                    CAPITAL_HAIN_DETAILS.NO_OF_SHARES
                    )
                ) AS ACTUAL_RATE, 
                SUM (
                    CAPITAL_HAIN_DETAILS.SALE_AMOUNT
                ) AS SALE_AMOUNT, 
                (
                    SUM (
                    CAPITAL_HAIN_DETAILS.PURCHASE_AMOUNT
                    ) / SUM (
                    CAPITAL_HAIN_DETAILS.NO_OF_SHARES
                    )
                ) AS PURCHASE_RATE, 
                SUM (
                    CAPITAL_HAIN_DETAILS.PURCHASE_AMOUNT
                ) AS PURCHASE_AMOUNT, 
                SUM (
                    CAPITAL_HAIN_DETAILS.CAPITAL_GAIN
                ) AS CAPITAL_GAIN 
            FROM 
            (
                SELECT 
                    GAIN_LOSS_DET.PORTFOLIO_CODE, 
                    GAIN_LOSS_DET.INVESTOR_NAME, 
                    GAIN_LOSS_DET.INSTRUMENT_DETAILS_ID, 
                    GAIN_LOSS_DET.INSTRUMENT_NAME, 
                    NVL (GAIN_LOSS_DET.SHARES, 0) AS NO_OF_SHARES, 
                    (
                        GAIN_LOSS_DET.TOTAL_AMOUNT - GAIN_LOSS_DET.BROKER_COMMISSION
                    ) / GAIN_LOSS_DET.SHARES AS ACTUAL_RATE, 
                    GAIN_LOSS_DET.TOTAL_AMOUNT - GAIN_LOSS_DET.BROKER_COMMISSION AS SALE_AMOUNT, 
                    GAIN_LOSS_DET.AVG_RATE AS PURCHASE_RATE, 
                    GAIN_LOSS_DET.SHARES * GAIN_LOSS_DET.AVG_RATE AS PURCHASE_AMOUNT, 
                    NVL (
                        (
                        GAIN_LOSS_DET.SHARES * GAIN_LOSS_DET.ACTUAL_RATE
                        ) - (
                        GAIN_LOSS_DET.SHARES * GAIN_LOSS_DET.AVG_RATE
                        ) - GAIN_LOSS_DET.BROKER_COMMISSION, 
                        0
                    ) AS CAPITAL_GAIN 
                FROM 
                (
                    SELECT 
                        INDV.PORTFOLIO_CODE, 
                        COALESCE (
                            INDV.INSTITUTION_NAME, INDV.INVESTOR_NAME
                        ) AS INVESTOR_NAME, 
                        IS_CAP_MKT_STOCK.INSTRUMENT_DETAILS_ID, 
                        LSID.INSTRUMENT_NAME, 
                        IS_CAP_MKT_STOCK.SHARES, 
                        (
                            CASE WHEN IS_CAP_MKT_STOCK.BUSINESS_DATE <= '31-Dec-2022' THEN (
                            IS_CAP_MKT_STOCK.BROKER_COMMISSION + IS_CAP_MKT_STOCK.SATTLEMENT_FEE
                            ) ELSE IS_CAP_MKT_STOCK.BROKER_COMMISSION END
                        ) AS BROKER_COMMISSION, 
                        IS_CAP_MKT_STOCK.AVG_RATE, 
                        IS_CAP_MKT_STOCK.ACTUAL_RATE, 
                        IS_CAP_MKT_STOCK.TOTAL_AMOUNT, 
                        IS_CAP_MKT_STOCK.BUSINESS_DATE, 
                        IBRKCHG.CHARGE 
                    FROM 
                        IS_CAP_MKT_STOCK, 
                        LS_INSTRUMENT_DETAILS LSID, 
                        IS_INVESTOR_DETAILS INDV, 
                        IS_I_BROKER_CHARGE IBRKCHG, 
                        (
                            SELECT 
                            MWINST.INSTRUMENT_DETAILS_ID, 
                            MWINST.SYMBOL 
                            FROM 
                            LS_MARKET_WISE_INSTRUMENT MWINST 
                            WHERE 
                            MWINST.MARKET_DETAILS_ID = 61
                        ) MWINSTRUMENT 
                    WHERE 
                        LSID.INSTRUMENT_DETAILS_ID = IS_CAP_MKT_STOCK.INSTRUMENT_DETAILS_ID 
                        AND IS_CAP_MKT_STOCK.PORTFOLIO_CODE = IBRKCHG.PORTFOLIO_CODE 
                        AND IBRKCHG.PORTFOLIO_CODE = INDV.PORTFOLIO_CODE 
                        AND INDV.IS_TEMPLATE = 'n' 
                        AND INDV.PORTFOLIO_CODE = :portfolio_code
                        AND IS_CAP_MKT_STOCK.I_BROKER_CHARGE_ID = IBRKCHG.I_BROKER_CHARGE_ID 
                        AND IS_CAP_MKT_STOCK.STATUS = 'enc' 
                        AND IS_CAP_MKT_STOCK.STOCK_MATURE_DATE IS NULL 
                        AND IS_CAP_MKT_STOCK.BUSINESS_DATE BETWEEN :from_date AND :to_date
                        AND MWINSTRUMENT.INSTRUMENT_DETAILS_ID = LSID.INSTRUMENT_DETAILS_ID
                ) GAIN_LOSS_DET
            ) CAPITAL_HAIN_DETAILS 
            GROUP BY 
                CAPITAL_HAIN_DETAILS.PORTFOLIO_CODE, 
                CAPITAL_HAIN_DETAILS.INVESTOR_NAME, 
                CAPITAL_HAIN_DETAILS.INSTRUMENT_DETAILS_ID, 
                CAPITAL_HAIN_DETAILS.INSTRUMENT_NAME 
            ORDER BY 
                CAPITAL_HAIN_DETAILS.INSTRUMENT_NAME ASC
        ";

        $gainLossSqlResult = DB::select(
            $gainLossSql,
            [
                'from_date'      => $fromDate->format('Y-m-d'),
                'to_date'        => $toDate->format('Y-m-d'),
                'portfolio_code' => $portfolioCode,
            ]
        );

        return $gainLossSqlResult;
    }
}
