<?php

namespace App\Http\Controllers\SellInfo;

use Exception;
use Carbon\Carbon;
use App\Models\BusinessDate;
use Illuminate\Http\Request;
use App\Models\CapMarketOrder;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use App\Models\CapMarketOrderDetails;
use Illuminate\Support\Facades\Validator;

class SellOrderController extends Controller
{
    public function instrumentList()
    {
        $user = Auth::user();
        $businessDate   = BusinessDate::latest('business_date')->first();

        $instrumentListSql = "
            SELECT 
                ISCMS.INSTRUMENT_DETAILS_ID, 
                ISCMS.PORTFOLIO_CODE, 
                ISCMS.TOTAL_SHARE, 
                ISCMS.MATURED_SHARE, 
                ISCMS.AVG_RATE, 
                HIS.CLOSE_PRICE, 
                INDV.IS_TEMPLATE, 
                LSID.INSTRUMENT_NAME, 
                SE.SECTOR_NAME, 
                (
                    SELECT 
                    DISTINCT LS_MARKET_WISE_INSTRUMENT.SYMBOL 
                    FROM 
                    LS_MARKET_WISE_INSTRUMENT 
                    WHERE 
                    LS_MARKET_WISE_INSTRUMENT.INSTRUMENT_DETAILS_ID = LSID.INSTRUMENT_DETAILS_ID
                ) AS SYMBOL 
            FROM 
                IS_CAP_MKT_STOCK ISCMS, 
                LS_INSTRUMENT_DETAILS LSID, 
                P_CAP_MKT_PRICE_HISTORY HIS, 
                IS_I_BROKER_CHARGE IBRKCHG, 
                IS_INVESTOR_DETAILS INDV, 
                GS_SECTOR SE 
            WHERE 
                HIS.INSTRUMENT_DETAILS_ID = ISCMS.INSTRUMENT_DETAILS_ID 
                AND HIS.INSTRUMENT_DETAILS_ID = LSID.INSTRUMENT_DETAILS_ID 
                AND ISCMS.I_BROKER_CHARGE_ID = IBRKCHG.I_BROKER_CHARGE_ID 
                AND INDV.PORTFOLIO_CODE = ISCMS.PORTFOLIO_CODE 
                AND ISCMS.PORTFOLIO_CODE = IBRKCHG.PORTFOLIO_CODE 
                AND SE.SECTOR_ID = LSID.SECTOR_ID 
                AND ISCMS.TOTAL_SHARE > 0 
                AND ISCMS.CAP_MKT_STOCK_ID = (
                    SELECT 
                    MAX (ISCMSD.CAP_MKT_STOCK_ID) 
                    FROM 
                    IS_CAP_MKT_STOCK ISCMSD 
                    WHERE 
                    ISCMSD.INSTRUMENT_DETAILS_ID = ISCMS.INSTRUMENT_DETAILS_ID 
                    AND ISCMS.PORTFOLIO_CODE = ISCMSD.PORTFOLIO_CODE 
                    AND LSID.INSTRUMENT_DETAILS_ID = ISCMSD.INSTRUMENT_DETAILS_ID 
                    AND HIS.INSTRUMENT_DETAILS_ID = ISCMSD.INSTRUMENT_DETAILS_ID 
                    AND ISCMSD.PORTFOLIO_CODE = :investor_code 
                    AND ISCMSD.BUSINESS_DATE < = :business_date
                ) 
                AND HIS.CAP_MKT_PRICE_HISTORY_ID = (
                    SELECT 
                    MAX (
                        P_CAP_MKT_PRICE_HISTORY.CAP_MKT_PRICE_HISTORY_ID
                    ) 
                    FROM 
                    P_CAP_MKT_PRICE_HISTORY 
                    WHERE 
                    P_CAP_MKT_PRICE_HISTORY.INSTRUMENT_DETAILS_ID = LSID.INSTRUMENT_DETAILS_ID 
                    AND P_CAP_MKT_PRICE_HISTORY.BUSINESS_DATE = (
                        SELECT 
                        MAX (
                            P_CAP_MKT_PRICE_HISTORY.BUSINESS_DATE
                        ) 
                        FROM 
                        P_CAP_MKT_PRICE_HISTORY 
                        WHERE 
                        P_CAP_MKT_PRICE_HISTORY.INSTRUMENT_DETAILS_ID = LSID.INSTRUMENT_DETAILS_ID 
                        AND P_CAP_MKT_PRICE_HISTORY.BUSINESS_DATE < = :business_date
                    )
                ) 
            ORDER BY 
            SYMBOL ASC, 
            ISCMS.PORTFOLIO_CODE ASC
        ";

        $instrumentListResult = DB::select(
            $instrumentListSql,
            [
                'business_date' => $businessDate->business_date,
                'investor_code' => $user->investor_code,
            ]
        );

        return response()->json([
            'status'          => 'success',
            'instrument_list' => $instrumentListResult,
        ]);
    }

    public function orderedList()
    {
        $user = Auth::user();
        $businessDate   = BusinessDate::latest('business_date')->first();
        $buyOrderList = CapMarketOrderDetails::with('instrumentDetails')
        ->where('buy_sell_flag', 'S')
        ->where('business_date', '<=', $businessDate->business_date)
        ->where('trade_code', $user->investor_code)
        ->get();

        $orderedList = [];

        foreach ($buyOrderList as $key => $list) {
            $orderedList[$key]['inst_id']          = $list['inst_id'];
            $orderedList[$key]['instrument_name']  = $list->instrumentDetails['instrument_name'];
            $orderedList[$key]['total_shares']     = $list['total_shares'];
            $orderedList[$key]['min_price']        = $list['max_min_price'];
            $orderedList[$key]['max_price']        = $list['max_min_price_to'];
            $orderedList[$key]['min_total_amount'] = $list['total_amount'];
            $orderedList[$key]['max_total_amount'] = $list['total_amount_to'];
            $orderedList[$key]['trade_time']       = is_null($list['trade_time']) ? null : Carbon::createFromFormat('Y-m-d H:i:s', $list['trade_time'])->format('d-M-Y');
            $orderedList[$key]['business_date']    = is_null($list['business_date']) ? null : Carbon::createFromFormat('Y-m-d H:i:s', $list['business_date'])->format('d-M-Y');
        }

        return response()->json([
            'status'          => 'success',
            'ordered_list' => $orderedList,
        ]);
    }

    public function submitOrder(Request $request)
    {
        try {
            $validator = Validator::make(
                $request->all(),
                [
                    'instrument_id.*' => 'required|numeric',
                    'price_from.*'    => "required|numeric",
                    'price_to.*'      => 'required|numeric',
                    'quantity.*'      => 'required|numeric',
                    'days.*'          => 'required|numeric'
                ],
                [
                    'instrument_id.*.required' => 'The instrument field is required',
                    'instrument_id.*.numeric' => 'The instrument field must be a number',
                    'price_from.*.required'    => "The price from field is required",
                    'price_from.*.numeric'    => "The price from field must be a number",
                    'price_to.*.required'    => "The price to field is required",
                    'price_to.*.numeric'    => "The price to field must be a number",
                    'quantity.*.required'    => "The quantity field is required",
                    'quantity.*.numeric'    => "The quantity field must be a number",
                    'days.*.required'    => "The days field is required",
                    'days.*.numeric'    => "The days field must be a number",
                ]
            );

            if (!$validator->fails()) {
                $processedData        = [];
                $submittedIstrumentId = [];
                $user                 = Auth::user();
                $requestedData        = $request->all();
                $businessDate         = BusinessDate::latest('business_date')->first();
                $instrumentList       = collect($this->instrumentList()->getData()->instrument_list);
                
                
                foreach ($requestedData['instrument_id'] as $key => $value) {
                    $submittedIstrumentId[]               = (int)$requestedData['instrument_id'][$key];
                    $processedData[$key]['instrument_id'] = (int)$requestedData['instrument_id'][$key];
                    $processedData[$key]['price_from']    = (float)$requestedData['price_from'][$key];
                    $processedData[$key]['price_to']      = (float)$requestedData['price_to'][$key];
                    $processedData[$key]['quantity']      = (int)$requestedData['quantity'][$key];
                    $processedData[$key]['days']          = (int)$requestedData['days'][$key];
                }
        
                
                // dd($instrumentList);
        
                foreach ($processedData as $key => $listData) {
                    $saleableShare = $instrumentList->where('instrument_details_id', $listData['instrument_id'])->first()?->matured_share;
                    
                    if (is_null($saleableShare) || ($listData['quantity'] > (int)$saleableShare)) {
                        throw new Exception('Sale quantity can not be greater than matured share quantity');
                    }
                }
        
                foreach ($processedData as $key => $listData) {
                    $capMarketOrderSql  = "SELECT nvl(max(web_id),0)+1 as web_id FROM is_cap_mkt_order";
                    $capMarketOrderData = collect(DB::select($capMarketOrderSql))->first();
                    
                    if (!is_null($capMarketOrderData)) {
                        $web_id                        = $capMarketOrderData->web_id;
                        $capMarketOrder                = new CapMarketOrder();
                        $capMarketOrder->web_id        = $web_id;
                        $capMarketOrder->payment_flag  = 'y';
                        $capMarketOrder->business_date = $businessDate->business_date;
                        $capMarketOrder->record_date   = now();

                        if ($capMarketOrder->save()) {
                            $capMarketOrderId                        = $capMarketOrder->cap_mkt_order_id;
                            $capMarketOrderDetails                   = new CapMarketOrderDetails();
                            $capMarketOrderDetails->cap_mkt_order_id = $capMarketOrderId;
                            $capMarketOrderDetails->trade_code       = $user->investor_code;
                            $capMarketOrderDetails->bd_id            = 681;
                            $capMarketOrderDetails->buy_sell_flag    = 'S';
                            $capMarketOrderDetails->business_date    = $businessDate->business_date;
                            $capMarketOrderDetails->trade_time       = now();
                            $capMarketOrderDetails->inst_id          = $listData['instrument_id'];
                            $capMarketOrderDetails->total_shares     = $listData['quantity'];
                            $capMarketOrderDetails->max_min_price    = $listData['price_from'];
                            $capMarketOrderDetails->max_min_price_to = $listData['price_to'];
                            $capMarketOrderDetails->total_amount     = ($listData['price_from'] * $listData['quantity']);
                            $capMarketOrderDetails->total_amount_to  = ($listData['price_to'] * $listData['quantity']);
                            $capMarketOrderDetails->days             = $listData['days'];
            
                            $capMarketOrderDetails->save();
                            
                        }
                    }
                }
        
                return response()->json([
                    'status'      => 'success',
                    'message'   => 'Sell order submitted successfully',
                ]);
            } else {
                $errors = [];

                foreach ($validator->errors()->messages() as $fieldName => $messages) {
                    $fieldNamePieces = explode('.', $fieldName);
                    // $errors[$fieldNamePieces[0]][$fieldNamePieces[1]] = $messages[0];
                    $errors[$fieldNamePieces[1]][$fieldNamePieces[0]] = $messages[0];
                }

                ksort($errors);

                return response()->json([
                    'status'  => 'error',
                    'message' => 'Form validation error',
                    'errors' => $errors,
                ], 400);
            }
        } catch (\Throwable $th) {
            return response()->json([
                'status'      => 'error',
                'message'   => $th->getMessage(),
            ], 400);
        }
    }

    public function pricePercentage()
    {
        return response()->json([
            'status'      => 'success',
            'percentage_value' => 10,
        ]);
    }
}
