<?php

namespace App\Http\Controllers\BuyInfo;

use Carbon\Carbon;
use App\Models\BusinessDate;
use Illuminate\Http\Request;
use App\Models\CapMarketOrder;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use App\Models\CapMarketOrderDetails;
use Illuminate\Support\Facades\Validator;

class BuyOrderController extends Controller
{
    public function instrumentList()
    {
        $instrumentListSql = "
            SELECT DISTINCT LSID.INSTRUMENT_DETAILS_ID,
                    HIS.CLOSE_PRICE,
                    LSID.INSTRUMENT_NAME,
                    SE.SECTOR_NAME,
                    SYM.SYMBOL
            FROM LS_INSTRUMENT_DETAILS LSID,
            GS_SECTOR SE,
            LS_MARKET_WISE_INSTRUMENT SYM,
            (SELECT X.INSTRUMENT_DETAILS_ID, X.CLOSE_PRICE
                FROM P_CAP_MKT_PRICE_HISTORY X,
                    (  SELECT INSTRUMENT_DETAILS_ID,
                                MAX (CAP_MKT_PRICE_HISTORY_ID)
                                    CAP_MKT_PRICE_HISTORY_ID
                            FROM P_CAP_MKT_PRICE_HISTORY
                        GROUP BY INSTRUMENT_DETAILS_ID) Y
            WHERE X.INSTRUMENT_DETAILS_ID    = Y.INSTRUMENT_DETAILS_ID
            AND   X.CAP_MKT_PRICE_HISTORY_ID = Y.CAP_MKT_PRICE_HISTORY_ID) HIS
            WHERE LSID.SECTOR_ID             = SE.SECTOR_ID(+)
            AND   LSID.INSTRUMENT_DETAILS_ID = SYM.INSTRUMENT_DETAILS_ID(+)
            AND   LSID.INSTRUMENT_DETAILS_ID = HIS.INSTRUMENT_DETAILS_ID
        ";

        $instrumentListResult = DB::select($instrumentListSql);

        return response()->json([
            'status'          => 'success',
            'instrument_list' => $instrumentListResult,
        ]);
    }

    public function orderedList()
    {
        $user         = Auth::user();
        $businessDate = BusinessDate::latest('business_date')->first();
        $buyOrderList = CapMarketOrderDetails::with('instrumentDetails')
        ->where('buy_sell_flag', 'B')
        ->where('business_date', $businessDate->business_date)
        ->where('trade_code', $user->investor_code)
        ->get();

        $orderedList = [];

        foreach ($buyOrderList as $key => $list) {
            $orderedList[$key]['inst_id']          = $list['inst_id'];
            $orderedList[$key]['instrument_name']  = $list->instrumentDetails['instrument_name'];
            $orderedList[$key]['total_shares']     = $list['total_shares'];
            $orderedList[$key]['min_price']        = $list['max_min_price'];
            $orderedList[$key]['max_price']        = $list['max_min_price_to'];
            $orderedList[$key]['min_total_amount'] = $list['total_amount'];
            $orderedList[$key]['max_total_amount'] = $list['total_amount_to'];
            $orderedList[$key]['trade_time']       = is_null($list['trade_time']) ? null : Carbon::createFromFormat('Y-m-d H:i:s', $list['trade_time'])->format('d-M-Y');
            $orderedList[$key]['business_date']    = is_null($list['business_date']) ? null : Carbon::createFromFormat('Y-m-d H:i:s', $list['business_date'])->format('d-M-Y');
        }

        return response()->json([
            'status'       => 'success',
            'ordered_list' => $orderedList,
        ]);
    }

    public function submitOrder(Request $request)
    {
        try {
            $validator = Validator::make(
                $request->all(),
                [
                    'instrument_id.*' => 'required|numeric',
                    'price_from.*'    => "required|numeric",
                    'price_to.*'      => 'required|numeric',
                    'quantity.*'      => 'required|numeric',
                    'days.*'          => 'required|numeric'
                ],
                [
                    'instrument_id.*.required' => 'The instrument field is required',
                    'instrument_id.*.numeric' => 'The instrument field must be a number',
                    'price_from.*.required'    => "The price from field is required",
                    'price_from.*.numeric'    => "The price from field must be a number",
                    'price_to.*.required'    => "The price to field is required",
                    'price_to.*.numeric'    => "The price to field must be a number",
                    'quantity.*.required'    => "The quantity field is required",
                    'quantity.*.numeric'    => "The quantity field must be a number",
                    'days.*.required'    => "The days field is required",
                    'days.*.numeric'    => "The days field must be a number",
                ]
            );

            // dd($validator->fails(), $validator->errors()->messages(), $validator->errors());

            if (!$validator->fails()) {
                $processedData = [];
                $user          = Auth::user();
                $requestedData = $request->all();
                $businessDate  = BusinessDate::latest('business_date')->first();
                $purchasepower = $this->purchasePower()->getData()->purchase_power;
        
                // return response()->json($requestedData['instrument_id']);
                // dd($requestedData['instrument_id']);
                
                foreach ($requestedData['instrument_id'] as $key => $value) {
                    $processedData[$key]['instrument_id'] = (int)$requestedData['instrument_id'][$key];
                    $processedData[$key]['price_from']    = (float)$requestedData['price_from'][$key];
                    $processedData[$key]['price_to']      = (float)$requestedData['price_to'][$key];
                    $processedData[$key]['quantity']      = (int)$requestedData['quantity'][$key];
                    $processedData[$key]['days']          = (int)$requestedData['days'][$key];
                }
        
                // return response()->json($processedData);
                $totalAmount = 0;
        
                foreach ($processedData as $key => $listData) {
                    $totalAmount += ($listData['price_to'] * $listData['quantity']);
                }
        
                if ($purchasepower >= $totalAmount) {
                    foreach ($processedData as $key => $listData) {                
                        $capMarketOrderSql  = "SELECT nvl(max(web_id),0)+1 as web_id FROM is_cap_mkt_order";
                        $capMarketOrderData = collect(DB::select($capMarketOrderSql))->first();
                        
                        if (!is_null($capMarketOrderData)) {
                            $web_id                        = $capMarketOrderData->web_id;
                            $capMarketOrder                = new CapMarketOrder();
                            $capMarketOrder->web_id        = $web_id;
                            $capMarketOrder->payment_flag  = 'y';
                            $capMarketOrder->business_date = $businessDate->business_date;
                            $capMarketOrder->record_date   = now();
        
                            if ($capMarketOrder->save()) {
                                $capMarketOrderId                        = $capMarketOrder->cap_mkt_order_id;
                                $capMarketOrderDetails                   = new CapMarketOrderDetails();
                                $capMarketOrderDetails->cap_mkt_order_id = $capMarketOrderId;
                                $capMarketOrderDetails->trade_code       = $user->investor_code;
                                $capMarketOrderDetails->bd_id            = 681;
                                $capMarketOrderDetails->buy_sell_flag    = 'B';
                                $capMarketOrderDetails->business_date    = $businessDate->business_date;
                                $capMarketOrderDetails->trade_time       = now();
                                $capMarketOrderDetails->inst_id          = $listData['instrument_id'];
                                $capMarketOrderDetails->total_shares     = $listData['quantity'];
                                $capMarketOrderDetails->max_min_price    = $listData['price_from'];
                                $capMarketOrderDetails->max_min_price_to = $listData['price_to'];
                                $capMarketOrderDetails->total_amount     = ($listData['price_from'] * $listData['quantity']);
                                $capMarketOrderDetails->total_amount_to  = ($listData['price_to'] * $listData['quantity']);
                                $capMarketOrderDetails->days             = $listData['days'];
                
                                $capMarketOrderDetails->save();
                                
                            }
                        }
                    }
                    
                    return response()->json([
                        'status'  => 'success',
                        'message' => 'Buy order submitted successfully',
                    ]);
                } else {
                    return response()->json([
                        'status'  => 'error',
                        'message' => 'Not available purchase power',
                    ], 400);
                }
            } else {
                $errors = [];
                foreach ($validator->errors()->messages() as $fieldName => $messages) {
                    $fieldNamePieces = explode('.', $fieldName);
                    $errors[$fieldNamePieces[1]][$fieldNamePieces[0]] = $messages[0];
                    // $errors[$fieldNamePieces[0]][$fieldNamePieces[1]] = $messages[0];
                }

                ksort($errors);

                return response()->json([
                    'status'  => 'error',
                    'message' => 'Form validation error',
                    'errors' => $errors,
                ], 400);
            }
            
        } catch (\Throwable $th) {
            return response()->json([
                'status'  => 'error',
                'message' => $th->getMessage(),
            ], 400);
        }
    }

    private function purchasePower()
    {
        $user         = Auth::user();
        $businessDate = BusinessDate::latest('business_date')->first();

        $purchasepower = DB::table('p_investor_portfolio_statement')
            ->selectRaw('distinct nvl(purchase_power, 0) as purchase_power')
            ->where('business_date', '=', $businessDate->business_date)
            ->where('portfolio_code', '=', $user->investor_code)
            ->first();

          // dd($purchasepower);

            
        return response()->json([
            'status'         => 'success',
            'purchase_power' => is_null($purchasepower) ? 999999999 : $purchasepower->purchase_power,
        ]);
    }

    public function pricePercentage()
    {
        return response()->json([
            'status'           => 'success',
            'percentage_value' => 10,
        ]);
    }
}
